@file:JvmName("MoshiSerializer")

package io.milkcan.effortlessprefs.moshiserializer

import android.util.Log
import com.squareup.moshi.Moshi
import io.milkcan.effortlessprefs.library.PrefSerializer
import java.io.IOException
import java.lang.reflect.Type

/**
 * @author Eric Bachhuber
 * @version 2.0.0
 * @since 1.1.0
 */
class MoshiSerializer(private val moshi: Moshi) : PrefSerializer() {

    companion object {
        @JvmStatic val TAG: String = MoshiSerializer::class.java.simpleName
    }

    /**
     * Stores an Object using Moshi.
     *
     * @param key The name of the preference to modify.
     * @param type Type of object to persist.
     * @param value The new value for the preference.
     */
    override fun <T> putObject(key: String, value: T, type: Type) {
        val adapter = moshi.adapter<T>(type::class.java)
        val json = adapter.toJson(value)

        preferences.edit().putString(key, json).apply()
    }

    /**
     * Retrieves a stored Object.
     *
     * @param key The name of the preference to retrieve.
     * @param defaultValue Value to return if this preference does not exist.
     * @param type Type of object to retrieve.
     * @return Deserialized representation of the object at [key], or [defaultValue] if unavailable
     * or an [IOException] is thrown while deserializing.
     */
    override fun <T> getObject(key: String, defaultValue: T, type: Type): T {
        val adapter = moshi.adapter<T>(type::class.java)
        val json = preferences.getString(key, "")

        return try {
            adapter.fromJson(json) ?: defaultValue
        } catch (ex: IOException) {
            Log.d(TAG, "Error deserializing object, returning default value. ${ex.message}", ex)
            defaultValue
        }
    }

    /**
     * Retrieves a stored Object.
     *
     * @param key The name of the preference to retrieve.
     * @param type Type of object to retrieve.
     * @return Deserialized representation of the object at [key], or null if unavailable or an
     * [IOException] is thrown while deserializing.
     */
    override fun <T> getObject(key: String, type: Type): T? {
        val adapter = moshi.adapter<T>(type)
        val json = preferences.getString(key, "")

        return try {
            adapter.fromJson(json)
        } catch (ex: IOException) {
            Log.d(TAG, "Error deserializing object, returning null. ${ex.message}", ex)
            null
        }
    }

}
