/*
 * Copyright 2017-2020 original authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.micronaut.retry.intercept;

import io.micronaut.core.annotation.Internal;
import io.micronaut.retry.RetryState;
import io.micronaut.retry.annotation.DefaultRetryPredicate;
import io.micronaut.retry.annotation.RetryPredicate;

import java.time.Duration;
import java.util.Optional;
import java.util.OptionalDouble;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;

import static java.lang.StrictMath.pow;

/**
 * A context object for {@link io.micronaut.retry.annotation.Retryable} operations.
 *
 * @author graemerocher
 * @since 1.0
 */
class SimpleRetry implements RetryState, MutableRetryState {

    private final int maxAttempts;
    private final double multiplier;
    private final Duration delay;
    private final Duration maxDelay;
    private final AtomicInteger attemptNumber = new AtomicInteger(0);
    private final AtomicLong overallDelay = new AtomicLong(0);
    private final RetryPredicate predicate;
    private final Class<? extends Throwable> capturedException;
    private final double jitter;

    /**
     * @param maxAttempts The maximum number of attempts
     * @param multiplier The multiplier to use between delays
     * @param delay The overall delay so far
     * @param maxDelay The maximum overall delay
     * @param predicate Predicate to check retry necessity
     * @param capturedException The capture exception types
     */
    SimpleRetry(
        int maxAttempts,
        double multiplier,
        Duration delay,
        Duration maxDelay,
        RetryPredicate predicate,
        Class<? extends Throwable> capturedException,
        double jitter) {

        this.maxAttempts = maxAttempts;
        this.multiplier = multiplier;
        this.delay = delay;
        this.maxDelay = maxDelay;
        this.predicate = predicate;
        this.capturedException = capturedException;
        this.jitter = jitter;
    }

    /**
     * @param maxAttempts The maximum number of attempts
     * @param multiplier The multiplier to use between delays
     * @param delay The overall delay so far
     * @param maxDelay The maximum overall delay
     * @param capturedException The capture exception types
     * @param jitter The jitter factor used to apply random deviation to retry delays
     */
    SimpleRetry(int maxAttempts, double multiplier, Duration delay, Duration maxDelay, Class<? extends Throwable> capturedException, double jitter) {
        this(maxAttempts, multiplier, delay, maxDelay, new DefaultRetryPredicate(), capturedException, jitter);
    }

    /**
     * @param maxAttempts The maximum number of attempts
     * @param multiplier The multiplier to use between delays
     * @param delay The overall delay so far
     * @param jitter The jitter factor used to apply random deviation to retry delays
     */
    SimpleRetry(int maxAttempts, double multiplier, Duration delay, double jitter) {
        this(maxAttempts, multiplier, delay, null, null, jitter);
    }

    /**
     * Should a retry attempt to occur.
     *
     * @return True if it should
     */
    @Override
    public boolean canRetry(Throwable exception) {
        if (exception == null) {
            return false;
        }

        if (!predicate.test(exception)) {
            return false;
        } else {
            return this.attemptNumber.incrementAndGet() < (maxAttempts + 1) && ((maxDelay == null) || overallDelay.get() < maxDelay.toMillis());
        }
    }

    /**
     * @return The maximum number of attempts
     */
    @Override
    public int getMaxAttempts() {
        return maxAttempts;
    }

    /**
     * @return The number of the current attempt
     */
    @Override
    public int currentAttempt() {
        return this.attemptNumber.get();
    }

    /**
     * @return The multiplier to use between delays
     */
    @Override
    public OptionalDouble getMultiplier() {
        return multiplier > 0 ? OptionalDouble.of(multiplier) : OptionalDouble.empty();
    }

    /**
     * @return The delay between attempts
     */
    @Override
    public Duration getDelay() {
        return delay;
    }

    /**
     * @return The overall delay so far
     */
    @Override
    public Duration getOverallDelay() {
        return Duration.ofMillis(overallDelay.get());
    }

    /**
     * @return The maximum overall delay
     */
    @Override
    public Optional<Duration> getMaxDelay() {
        return Optional.ofNullable(maxDelay);
    }

    @Override
    public RetryPredicate getRetryPredicate() {
        return predicate;
    }

    @Override
    public Class<? extends Throwable> getCapturedException() {
        return capturedException;
    }

    /**
     * @return The jitter factor used to apply random deviation to retry delays
     */
    @Override
    public OptionalDouble getJitter() {
        return jitter > 0 ? OptionalDouble.of(jitter) : OptionalDouble.empty();
    }

    /**
     * @return Return the millisecond value for the next delay
     */
    @Override
    @Internal
    public long nextDelay() {
        double multiplier = getMultiplier().orElse(1.0);
        long delay = (long) ((getDelay().toMillis()) * pow(multiplier, attemptNumber.get() - 1));
        double jitter = getJitter().orElse(0);
        if (jitter > 0) {
            delay = Math.max(0, (long) (delay * (1.0 + ThreadLocalRandom.current().nextDouble(-jitter, jitter))));
        }
        overallDelay.addAndGet(delay);
        return delay;
    }
}
