/*
 * Copyright 2017-2020 original authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.micronaut.health;

import io.micronaut.context.condition.Condition;
import io.micronaut.context.condition.ConditionContext;
import io.micronaut.core.annotation.Introspected;
import io.micronaut.core.annotation.ReflectiveAccess;
import io.micronaut.core.convert.ConversionContext;
import io.micronaut.discovery.CompositeDiscoveryClient;
import io.micronaut.discovery.DiscoveryClient;

import static java.lang.Boolean.FALSE;

/**
 * Custom condition to conditionally enable the heart beat.
 *
 * @author graemerocher
 * @since 1.1
 */
@Introspected
@ReflectiveAccess
public final class HeartbeatDiscoveryClientCondition implements Condition {
    @Override
    public boolean matches(ConditionContext context) {
        final boolean hasDiscovery = context.getBeanContext().getBeanDefinitions(DiscoveryClient.class).stream()
            .anyMatch(bd -> !CompositeDiscoveryClient.class.isAssignableFrom(bd.getBeanType()));
        if (hasDiscovery) {
            return true;
        }
        final Boolean enabled = context.getProperty(HeartbeatConfiguration.ENABLED, ConversionContext.BOOLEAN).orElse(FALSE);
        if (!enabled) {
            context.fail("Heartbeat not enabled since no Discovery client active");
        }
        return enabled;
    }
}
