package io.maxads.ads.interstitial.vast3.model;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import io.maxads.ads.base.Constants;

public class VASTVideoConfig implements Serializable {
  private static final long serialVersionUID = 0L;
  @NonNull private static final String TAG = VASTVideoConfig.class.getSimpleName();

  @NonNull private final List<VASTTracker> mImpressionTrackers;
  @NonNull private final List<VASTTracker> mErrorTrackers;
  @NonNull private final List<VASTTracker> mClickTrackers;
  @NonNull private final Map<VASTEventTracker.Event, List<VASTTracker>> mEventTrackers;
  @NonNull private final List<VASTAbsoluteTracker> mAbsoluteTrackers;
  @NonNull private final List<VASTFractionalTracker> mFractionalTrackers;
  @NonNull private VASTOffset mSkipOffset;
  private int mMaxDurationMs;
  @Nullable private String mClickThroughUrl;
  @Nullable private String mMediaFileUrl;
  @Nullable private String mMediaFileDiskUrl;
  @Nullable private VASTCompanionAdConfig mVASTCompanionAdConfig;
  @Nullable private VASTIconConfig mVASTIconConfig;
  
  // Video state
  private boolean mVideoStarted;
  private boolean mVideoCompleted;
  private boolean mVideoErrored;
  private boolean mVideoCloseable;
  private int mCurrentVideoPositionMs;

  public VASTVideoConfig() {
    mImpressionTrackers = new ArrayList<>();
    mErrorTrackers = new ArrayList<>();
    mClickTrackers = new ArrayList<>();
    mEventTrackers = new HashMap<>();
    mAbsoluteTrackers = new ArrayList<>();
    mFractionalTrackers = new ArrayList<>();

    // Ignore the skip offset from the VAST xml and instead default to 5 seconds
    // unless we receive a server side skip offset value
    mSkipOffset = VASTOffset.from(Constants.DEFAULT_VAST_SKIP_OFFSET_MS);
    mMaxDurationMs = Constants.DEFAULT_VAST_MAX_DURATION_MS;
  }

  public boolean isVideoStarted() {
    return mVideoStarted;
  }

  public void setVideoStarted(boolean videoStarted) {
    mVideoStarted = videoStarted;
  }

  public boolean isVideoCompleted() {
    return mVideoCompleted;
  }

  public void setVideoCompleted(boolean videoCompleted) {
    mVideoCompleted = videoCompleted;
  }

  public boolean isVideoErrored() {
    return mVideoErrored;
  }

  public void setVideoErrored(boolean videoErrored) {
    mVideoErrored = videoErrored;
  }

  public boolean isVideoCompletedOrErrored() {
    return mVideoCompleted || mVideoErrored;
  }

  public boolean isVideoCloseable() {
    return mVideoCloseable;
  }

  public void setVideoCloseable(boolean videoCloseable) {
    mVideoCloseable = videoCloseable;
  }

  public int getCurrentVideoPositionMs() {
    return mCurrentVideoPositionMs;
  }

  public void setCurrentVideoPositionMs(int currentVideoPositionMs) {
    mCurrentVideoPositionMs = currentVideoPositionMs;
  }

  @NonNull
  public List<VASTTracker> getImpressionTrackers() {
    return mImpressionTrackers;
  }

  public void addImpressionTrackers(@NonNull List<VASTTracker> vastTrackers) {
    mImpressionTrackers.addAll(vastTrackers);
  }

  @NonNull
  public List<VASTTracker> getErrorTrackers() {
    return mErrorTrackers;
  }

  public void addErrorTracker(@NonNull List<VASTTracker> vastTracker) {
    mErrorTrackers.addAll(vastTracker);
  }

  @NonNull
  public List<VASTTracker> getClickTrackers() {
    return mClickTrackers;
  }

  public void addClickTrackers(@NonNull List<VASTTracker> vastTrackers) {
    mClickTrackers.addAll(vastTrackers);
  }

  @NonNull
  public List<VASTTracker> getEventTrackers(@NonNull VASTEventTracker.Event event) {
    return mEventTrackers.containsKey(event) ? mEventTrackers.get(event) : Collections.<VASTTracker>emptyList();
  }

  @NonNull
  public List<VASTTracker> getAllEventTrackers() {
    final List<VASTTracker> allVASTTrackers = new ArrayList<>(mEventTrackers.size());
    for (List<VASTTracker> vastTrackers : mEventTrackers.values()) {
      allVASTTrackers.addAll(vastTrackers);
    }
    return allVASTTrackers;
  }

  public void addEventTrackers(@NonNull List<VASTEventTracker> vastTrackers) {
    for (VASTEventTracker vastTracker : vastTrackers) {
      addEventTracker(vastTracker);
    }
  }

  public void addEventTracker(@NonNull VASTEventTracker vastEventTracker) {
    final VASTEventTracker.Event event = vastEventTracker.getEvent();
    if (mEventTrackers.containsKey(event)) {
      mEventTrackers.get(event).add(vastEventTracker);
    } else {
      final ArrayList<VASTTracker> vastTrackers = new ArrayList<>(1);
      vastTrackers.add(vastEventTracker);
      mEventTrackers.put(event, vastTrackers);
    }
  }

  @NonNull
  public List<VASTAbsoluteTracker> getAbsoluteTrackers() {
    return mAbsoluteTrackers;
  }

  @NonNull
  public List<VASTFractionalTracker> getFractionalTrackers() {
    return mFractionalTrackers;
  }

  public void addVASTTrackers(@NonNull List<VASTTracker> vastTrackers) {
    for (VASTTracker vastTracker : vastTrackers) {
      if (vastTracker instanceof VASTEventTracker) {
        addEventTracker((VASTEventTracker) vastTracker);
      } else if (vastTracker instanceof VASTAbsoluteTracker) {
        mAbsoluteTrackers.add((VASTAbsoluteTracker) vastTracker);
      } else if (vastTracker instanceof VASTFractionalTracker) {
        mFractionalTrackers.add((VASTFractionalTracker) vastTracker);
      }
    }
  }

  @NonNull
  public VASTOffset getSkipOffset() {
    return mSkipOffset;
  }

  public void setSkipOffset(int skipOffsetMs) {
    mSkipOffset = VASTOffset.from(skipOffsetMs);
  }

  public int getMaxDurationMs() {
    return mMaxDurationMs;
  }

  public void setMaxDurationMs(int maxDurationMs) {
    mMaxDurationMs = maxDurationMs;
  }

  public int getNewVideoDurationMs(int videoDurationMs) {
    return mMaxDurationMs > 0 && mMaxDurationMs < videoDurationMs ? mMaxDurationMs : videoDurationMs;
  }

  public boolean maxDurationExceeded(int currentPositionMs) {
    return mMaxDurationMs > 0 && currentPositionMs >= mMaxDurationMs;
  }

  @Nullable
  public String getClickThroughUrl() {
    return mClickThroughUrl;
  }

  public void setClickThroughUrl(@Nullable String clickThroughUrl) {
    mClickThroughUrl = clickThroughUrl != null ? clickThroughUrl.trim() : null;
  }

  @Nullable
  public String getMediaFileUrl() {
    return mMediaFileUrl;
  }

  public void setMediaFileUrl(@Nullable String mediaFileUrl) {
    mMediaFileUrl = mediaFileUrl != null ? mediaFileUrl.trim() : null;
  }

  @Nullable
  public String getMediaFileDiskUrl() {
    return mMediaFileDiskUrl;
  }

  public void setMediaFileDiskUrl(@Nullable String mediaFileDiskUrl) {
    mMediaFileDiskUrl = mediaFileDiskUrl;
  }

  @Nullable
  public VASTCompanionAdConfig getVASTCompanionAdConfig() {
    return mVASTCompanionAdConfig;
  }

  public void setVASTCompanionAdConfig(@Nullable VASTCompanionAdConfig vastCompanionAdConfig) {
    mVASTCompanionAdConfig = vastCompanionAdConfig;
  }

  @Nullable
  public VASTIconConfig getVASTIconConfig() {
    return mVASTIconConfig;
  }

  public void setVASTIconConfig(@Nullable VASTIconConfig vastIconConfig) {
    mVASTIconConfig = vastIconConfig;
  }
}
