package io.maxads.ads.interstitial.vast3;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

import io.maxads.ads.base.util.MaxAdsLog;
import io.maxads.ads.interstitial.vast3.model.VASTVideoConfig;
import io.maxads.ads.interstitial.vast3.view.VASTViewModule;
import io.reactivex.Observable;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.disposables.Disposable;
import io.reactivex.functions.Consumer;

public class VASTTimer {
  @NonNull private static final String TAG = VASTTimer.class.getSimpleName();
  private int mVideoDurationMs;
  private boolean mMaxDurationExceeded;

  public interface Listener {
    void onTick(int videoDurationMs, int currentPositionMs);
    void onStopped(int videoDurationMs, boolean videoCompleted, boolean maxDurationExceeded);
    void onMaxDurationExceeded();
  }

  @NonNull private final VASTViewModule mVASTViewModule;
  @NonNull private final VASTVideoConfig mVASTVideoConfig;
  @NonNull private final List<Listener> mListeners;
  @Nullable private Disposable mTimerInterval;

  public VASTTimer(@NonNull VASTViewModule vastViewModule,
                   @NonNull VASTVideoConfig vastVideoConfig) {
    mVASTViewModule = vastViewModule;
    mVASTVideoConfig = vastVideoConfig;
    mListeners = new ArrayList<>();
  }

  public void addListener(@NonNull Listener listener) {
    mListeners.add(listener);
  }

  public void addListeners(@NonNull List<Listener> listeners) {
    mListeners.addAll(listeners);
  }

  public void startTimer() {
    if ((mTimerInterval != null && !mTimerInterval.isDisposed()) || mVASTVideoConfig.isVideoCompletedOrErrored()) {
      return;
    }

    mVideoDurationMs = mVASTViewModule.videoGetDurationMs();
    mTimerInterval = Observable.interval(0, 250, TimeUnit.MILLISECONDS)
      // Must be main thread since we are interacting with UI components
      .observeOn(AndroidSchedulers.mainThread())
      .subscribe(new Consumer<Long>() {
        @Override
        public void accept(Long aLong) {
          onTimerTick(mVideoDurationMs, aLong);
        }
      });
  }

  @VisibleForTesting
  void onTimerTick(int videoDurationMs, Long aLong) {
    final int currentPositionMs;
    try {
      currentPositionMs = mVASTViewModule.videoGetCurrentPositionMs();
    } catch (Exception e) {
      MaxAdsLog.w(TAG, "Exception getting video current position: " + e.getMessage(), e);
      return;
    }

    if (currentPositionMs <= 0) {
      return;
    }

    for (Listener listener : mListeners) {
      listener.onTick(videoDurationMs, currentPositionMs);
    }

    // Check for max duration exceeded after onTick so that we fire all trackers up to that point in time
    if (mVASTVideoConfig.maxDurationExceeded(currentPositionMs)) {
      mMaxDurationExceeded = true;
      for (Listener listener : mListeners) {
        listener.onMaxDurationExceeded();
      }
    }
  }

  public void stopTimer(boolean videoCompleted) {
    stopTimer(mVideoDurationMs, videoCompleted, mMaxDurationExceeded, mTimerInterval);
  }

  @VisibleForTesting
  void stopTimer(int videoDurationMs,
                 boolean videoCompleted,
                 boolean maxDurationExceeded,
                 @Nullable Disposable timerInterval) {
    for (Listener listener : mListeners) {
      listener.onStopped(videoDurationMs, videoCompleted, maxDurationExceeded);
    }

    if (timerInterval != null) {
      timerInterval.dispose();
    }
  }
}
