package io.maxads.ads.interstitial.vast3;

import android.support.annotation.NonNull;
import android.support.annotation.VisibleForTesting;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import io.maxads.ads.interstitial.vast3.model.VASTAbsoluteTracker;
import io.maxads.ads.interstitial.vast3.model.VASTFractionalTracker;

public class VASTProgressTimerListener implements VASTTimer.Listener {
  @NonNull private final VASTApiClientDecorator mVASTApiClientDecorator;
  @NonNull private final List<VASTAbsoluteTracker> mVASTAbsoluteTrackers;
  @NonNull private final List<VASTFractionalTracker> mVASTFractionalTrackers;

  public VASTProgressTimerListener(@NonNull VASTApiClientDecorator vastApiClientDecorator,
                                   @NonNull List<VASTAbsoluteTracker> vastAbsoluteTrackers,
                                   @NonNull List<VASTFractionalTracker> vastFractionalTrackers) {
    mVASTApiClientDecorator = vastApiClientDecorator;
    mVASTAbsoluteTrackers = new ArrayList<>(vastAbsoluteTrackers);
    Collections.sort(mVASTAbsoluteTrackers);
    mVASTFractionalTrackers = new ArrayList<>(vastFractionalTrackers);
    Collections.sort(mVASTFractionalTrackers);
  }

  @Override
  public void onTick(int videoDurationMs, int currentPositionMs) {
    fireTrackers(videoDurationMs, currentPositionMs);
  }

  @Override
  public void onStopped(int videoDuationMs, boolean videoCompleted, boolean maxDurationExceeded) {
    if (videoCompleted && !maxDurationExceeded) {
      fireTrackers(videoDuationMs, videoDuationMs);
    }
  }

  @Override
  public void onMaxDurationExceeded() {
  }

  @VisibleForTesting
  void fireTrackers(int videoDurationMs, int currentPositionMs) {
    if (videoDurationMs <= 0 || currentPositionMs <= 0) {
      return;
    }

    final Iterator<VASTAbsoluteTracker> iterator = mVASTAbsoluteTrackers.iterator();
    while (iterator.hasNext()) {
      final VASTAbsoluteTracker vastAbsoluteTracker = iterator.next();
      if (vastAbsoluteTracker.getTimeMs() <= currentPositionMs) {
        mVASTApiClientDecorator.fireVASTTracker(vastAbsoluteTracker);
        iterator.remove();
      } else {
        // List is sorted so we can exit early
        break;
      }
    }

    float fraction = (float) currentPositionMs / videoDurationMs;
    final Iterator<VASTFractionalTracker> iterator2 = mVASTFractionalTrackers.iterator();
    while (iterator2.hasNext()) {
      final VASTFractionalTracker vastFractionalTracker = iterator2.next();
      if (Float.compare(vastFractionalTracker.getFraction(), fraction) <= 0) {
        mVASTApiClientDecorator.fireVASTTracker(vastFractionalTracker);
        iterator2.remove();
      } else {
        // List is sorted so we can exit early
        break;
      }
    }
  }

  @VisibleForTesting
  @NonNull
  List<VASTAbsoluteTracker> getVASTAbsoluteTrackers() {
    return mVASTAbsoluteTrackers;
  }

  @VisibleForTesting
  @NonNull
  List<VASTFractionalTracker> getVASTFractionalTrackers() {
    return mVASTFractionalTrackers;
  }
}
