package io.maxads.ads.interstitial.vast3;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import android.util.DisplayMetrics;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import io.maxads.ads.base.util.MaxAdsLog;
import io.maxads.ads.base.util.Util;
import io.maxads.ads.interstitial.vast3.model.VASTTracker;
import io.maxads.ads.interstitial.vast3.xml_model.MediaFileXml;

public class MediaFilePicker {
  @NonNull private static final String TAG = MediaFilePicker.class.getSimpleName();

  // http://developer.android.com/guide/appendix/media-formats.html#core
  @NonNull private static final List<String> SUPPORTED_VIDEO_MIME_TYPES
    = Arrays.asList("video/mp4", "video/3gpp", "video/webm");

  @NonNull private final Context mContext;
  @NonNull private final AreaComparator mAreaComparator;

  public MediaFilePicker(@NonNull Context context) {
    mContext = context;

    final DisplayMetrics metrics = mContext.getResources().getDisplayMetrics();
    final int deviceWidth = metrics.widthPixels;
    final int deviceHeight = metrics.heightPixels;
    final int deviceArea = deviceWidth * deviceHeight;

    mAreaComparator = new AreaComparator(deviceArea);
  }

  @Nullable
  public MediaFileXml pickMediaFile(@NonNull VASTApiClientDecorator vastApiClientDecorator,
                                    @NonNull List<VASTTracker> vastErrorTrackers,
                                    @Nullable List<MediaFileXml> mediaFileXmls) {
    if (mediaFileXmls == null) {
      return null;
    }

    final List<MediaFileXml> validMediaFileXmls =
      prefilterMediaFiles(vastApiClientDecorator, vastErrorTrackers, mediaFileXmls);
    if (validMediaFileXmls.isEmpty()) {
      return null;
    }

    Collections.sort(validMediaFileXmls, mAreaComparator);
    return validMediaFileXmls.get(0);
  }

  @NonNull
  private List<MediaFileXml> prefilterMediaFiles(@NonNull VASTApiClientDecorator vastApiClientDecorator,
                                                 @NonNull List<VASTTracker> vastErrorTrackers,
                                                 @NonNull List<MediaFileXml> mediaFileXmls) {
    final List<MediaFileXml> validMediaFileXmls = new ArrayList<>(mediaFileXmls.size());
    for (MediaFileXml mediaFileXml : mediaFileXmls) {
      final String url = mediaFileXml.value;
      if (TextUtils.isEmpty(url)) {
        MaxAdsLog.d(TAG, "Invalid media file url: " + mediaFileXml);
        continue;
      }

      final String type = mediaFileXml.type;
      if (!SUPPORTED_VIDEO_MIME_TYPES.contains(type)) {
        MaxAdsLog.d(TAG, "Invalid media file type: " + type);
        vastApiClientDecorator.fireVASTTrackers(vastErrorTrackers,
            new VASTMacroDataImpl.Builder()
              .withAssetUri(mediaFileXml.value)
              .withVASTError(VASTError.INVALID_MEDIA_FILE_TYPE)
              .build());
        continue;
      }

      final Integer heightDp = mediaFileXml.height;
      if (heightDp == null || heightDp < 0) {
        MaxAdsLog.d(TAG, "Invalid media file height: " + heightDp);
        vastApiClientDecorator.fireVASTTrackers(vastErrorTrackers,
          new VASTMacroDataImpl.Builder()
            .withAssetUri(mediaFileXml.value)
            .withVASTError(VASTError.INCORRECT_SIZE)
            .build());
        continue;
      }

      final Integer widthDp = mediaFileXml.width;
      if (widthDp == null || widthDp < 0) {
        MaxAdsLog.d(TAG, "Invalid media file width: " + widthDp);
        vastApiClientDecorator.fireVASTTrackers(vastErrorTrackers,
          new VASTMacroDataImpl.Builder()
            .withAssetUri(mediaFileXml.value)
            .withVASTError(VASTError.INCORRECT_SIZE)
            .build());
        continue;
      }

      mediaFileXml.widthPx = Util.dipsToIntPixels(mediaFileXml.width, mContext);
      mediaFileXml.heightPx = Util.dipsToIntPixels(mediaFileXml.height, mContext);

      validMediaFileXmls.add(mediaFileXml);
    }

    return validMediaFileXmls;
  }

  private static class AreaComparator implements Comparator<MediaFileXml> {
    private final int mDeviceArea;

    private AreaComparator(int deviceArea) {
      mDeviceArea = deviceArea;
    }

    @Override
    public int compare(MediaFileXml obj1, MediaFileXml obj2) {
      // get area of the video of the two MediaFiles
      int obj1Area = obj1.widthPx * obj1.heightPx;
      int obj2Area = obj2.widthPx * obj2.heightPx;

      // get the difference between the area of the MediaFile and the area of the screen
      int obj1Diff = Math.abs(mDeviceArea - obj1Area);
      int obj2Diff = Math.abs(mDeviceArea - obj2Area);

      // choose the MediaFile which has the lower difference in area
      if(obj1Diff < obj2Diff) {
        return -1;
      } else if(obj1Diff > obj2Diff) {
        return 1;
      } else {
        return 0;
      }
    }
  }
}
