package io.maxads.ads.interstitial;

import android.app.Activity;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;

import java.util.Map;

import io.maxads.ads.base.MaxAds;
import io.maxads.ads.base.model.Ad;
import io.maxads.ads.base.model.Winner;
import io.maxads.ads.base.util.MaxAdsLog;

public class InterstitialDecoratorFactory {
  @NonNull private static final String TAG = InterstitialDecoratorFactory.class.getSimpleName();

  @NonNull private final MraidInterstitial.MraidInterstitialFactory mMraidInterstitialFactory;
  @NonNull private final VAST3Interstitial.VAST3InterstitialFactory mVast3InterstitialFactory;
  @NonNull private final Map<String, InterstitialFactory> mPartnerInterstitialFactories;

  public InterstitialDecoratorFactory(@NonNull Activity activity) {
    this(new MraidInterstitial.MraidInterstitialFactory(activity),
      new VAST3Interstitial.VAST3InterstitialFactory(activity), MaxAds.getPartnerInterstitialFactories());
  }

  @VisibleForTesting
  InterstitialDecoratorFactory(@NonNull MraidInterstitial.MraidInterstitialFactory mraidInterstitialFactory,
                               @NonNull VAST3Interstitial.VAST3InterstitialFactory vast3InterstitialFactory,
                               @NonNull Map<String, InterstitialFactory> partnerInterstitialFactories) {
    mMraidInterstitialFactory = mraidInterstitialFactory;
    mVast3InterstitialFactory = vast3InterstitialFactory;
    mPartnerInterstitialFactories = partnerInterstitialFactories;
  }

  @Nullable
  public InterstitialDecorator createInterstitial(@NonNull Ad ad, @NonNull InterstitialDecorator.Listener listener) {
    final Interstitial interstitial;
    final Winner winner = ad.getWinner();
    if (winner.usePartnerRendering()) {
      interstitial = fromPartnerInterstitialFactories(winner.getPartner(), ad);
    } else {
      interstitial = fromCreativeType(ad.getWinner().getCreativeType(), ad);
    }

    if (interstitial == null) {
      return null;
    }

    final InterstitialDecorator interstitialDecorator = new InterstitialDecorator(interstitial, ad);
    interstitialDecorator.setListener(listener);
    return interstitialDecorator;
  }

  @Nullable
  @VisibleForTesting
  Interstitial fromPartnerInterstitialFactories(@Nullable String partner, @NonNull Ad ad) {
    if (!mPartnerInterstitialFactories.containsKey(partner)) {
      MaxAdsLog.e(TAG, "Did not find a valid partner InterstitialFactory for partner: " + partner);
      return null;
    }

    return mPartnerInterstitialFactories.get(partner).createInterstitial(ad);
  }

  @Nullable
  @VisibleForTesting
  Interstitial fromCreativeType(@NonNull Winner.CreativeType creativeType, @NonNull Ad ad) {
    switch (creativeType) {
      case HTML: {
        return mMraidInterstitialFactory.createInterstitial(ad);
      }
      case VAST3: {
        return mVast3InterstitialFactory.createInterstitial(ad);
      }
      case EMPTY: {
        MaxAdsLog.d(TAG, "No ads found for for ad unit id: " + ad.getAdUnitId());
        return null;
      }
      default: {
        MaxAdsLog.e(TAG, "Incompatible creative type: " + creativeType + ", for interstitial ad format.");
        return null;
      }
    }
  }
}
