package io.maxads.ads.interstitial;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;
import android.support.v4.content.LocalBroadcastManager;

import java.util.Random;

public class InterstitialBroadcastReceiver extends BroadcastReceiver {

  public interface Listener {
    void onReceivedAction(@NonNull Action action);
  }

  public enum Action {
    SHOW("io.maxads.ads.interstitial.show"),
    IMPRESSED("io.maxads.ads.interstitial.impressed"),
    CLICK("io.maxads.ads.interstitial.click"),
    DISMISS("io.maxads.ads.interstitial.dismiss"),
    ERROR("io.maxads.ads.interstitial.error"),
    NONE("none");

    @NonNull
    public static Action from(@Nullable String action) {
      if (SHOW.getId().equals(action)) {
        return SHOW;
      } else if (IMPRESSED.getId().equals(action)) {
        return IMPRESSED;
      } else if (CLICK.getId().equals(action)) {
        return CLICK;
      } else if (DISMISS.getId().equals(action)) {
        return DISMISS;
      } else if (ERROR.getId().equals(action)) {
        return ERROR;
      }

      return NONE;
    }

    @NonNull private final String mId;

    Action(@NonNull String id) {
      mId = id;
    }

    @NonNull
    public String getId() {
      return mId;
    }
  }

  @NonNull public static final String BROADCAST_ID = "broadcastId";

  private final long mBroadcastId;
  @NonNull private final LocalBroadcastManager mLocalBroadcastManager;
  @NonNull private final IntentFilter mIntentFilter;
  private boolean mDestroyed;

  @Nullable private Listener mListener;

  public InterstitialBroadcastReceiver(@NonNull Context context) {
    this(new Random().nextLong(), LocalBroadcastManager.getInstance(context), new IntentFilter());
  }

  @VisibleForTesting
  InterstitialBroadcastReceiver(long broadcastId,
                                @NonNull LocalBroadcastManager localBroadcastManager,
                                @NonNull IntentFilter intentFilter) {
    mBroadcastId = broadcastId;
    mLocalBroadcastManager = localBroadcastManager;

    mIntentFilter = intentFilter;
    mIntentFilter.addAction(Action.SHOW.getId());
    mIntentFilter.addAction(Action.IMPRESSED.getId());
    mIntentFilter.addAction(Action.CLICK.getId());
    mIntentFilter.addAction(Action.DISMISS.getId());
    mIntentFilter.addAction(Action.ERROR.getId());
  }

  public void setListener(@Nullable Listener listener) {
    mListener = listener;
  }

  public long getBroadcastId() {
    return mBroadcastId;
  }

  public void register() {
    if (mDestroyed) {
      return;
    }

    mLocalBroadcastManager.registerReceiver(this, mIntentFilter);
  }

  public void destroy() {
    mLocalBroadcastManager.unregisterReceiver(this);
    mDestroyed = true;
  }

  @Override
  public void onReceive(Context context, Intent intent) {
    if (mDestroyed || mListener == null) {
      return;
    }

    final long receivedId = intent.getLongExtra(BROADCAST_ID, -1);
    if (mBroadcastId != receivedId) {
      return;
    }

    mListener.onReceivedAction(Action.from(intent.getAction()));
  }

  public void handleAction(@NonNull Action action,
                           @NonNull Interstitial interstitial,
                           @Nullable Interstitial.Listener listener) {
    if (listener == null) {
      return;
    }

    switch (action) {
      case SHOW:
        listener.onInterstitialShown(interstitial);
        break;
      case IMPRESSED:
        listener.onInterstitialImpressed(interstitial);
        break;
      case CLICK:
        listener.onInterstitialClicked(interstitial);
        break;
      case DISMISS:
        listener.onInterstitialDismissed(interstitial);
        break;
      case ERROR:
        listener.onInterstitialError(interstitial);
        break;
      case NONE:
        break;
    }
  }
}
