package io.maxads.ads.base.model;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import io.maxads.ads.base.MaxAds;
import io.maxads.ads.base.RefreshTimer;
import io.maxads.ads.base.api.AdResponse;
import io.maxads.ads.base.api.ApiClient;
import io.maxads.ads.base.util.Helpers;
import io.maxads.ads.base.util.MaxAdsLog;

public class Ad {
  @NonNull private static final String TAG = Ad.class.getSimpleName();

  private enum TrackingType {
    HANDOFF("handoff"),
    SELECTED("selected"),
    IMPRESSION("impression"),
    CLICK("click"),
    EXPIRE("expire"),
    LOSS("loss"),
    ERROR("error");

    @NonNull private final String mType;

    TrackingType(@NonNull String type) {
      mType = type;
    }

    @Override
    public String toString() {
      return mType;
    }
  }

  @NonNull private static final long DEFAULT_EXPIRATION_MS = TimeUnit.HOURS.toMillis(1);

  @NonNull private final ApiClient mApiClient;
  @NonNull private final String mAdUnitId;
  @Nullable private final String mCreative;
  @NonNull private final String mPrebidKeywords;
  private final int mRefreshTimeSeconds;
  @NonNull private final List<String> mHandoffUrls;
  @NonNull private final List<String> mSelectedUrls;
  @NonNull private final List<String> mImpressionUrls;
  @NonNull private final List<String> mClickUrls;
  @NonNull private final List<String> mExpireUrls;
  @NonNull private final List<String> mLossUrls;
  @NonNull private final List<String> mErrorUrls;
  // Prioritize this ad response for a specific MAX bidder
  private final boolean mReserved;
  private final long mExpirationTimeMs;
  private final long mCreatedAtMs;
  @NonNull private final Winner mWinner;
  @NonNull private final AdUnit mAdUnit;
  @NonNull private final Helpers.SystemTimeHelper mSystemTimeHelper;

  private boolean mHandoffTracked;
  private boolean mSelectedTracked;
  private boolean mImpressionTracked;
  private boolean mClickTracked;
  private boolean mExpireTracked;
  private boolean mLossTracked;

  public static Ad from(@NonNull String adUnitId, @NonNull AdResponse adResponse) {
    return from(MaxAds.getApiClient(), adUnitId, adResponse);
  }

  @VisibleForTesting
  static Ad from(@NonNull ApiClient apiClient, @NonNull String adUnitId, @NonNull AdResponse adResponse) {
    return new Ad(apiClient, adUnitId, adResponse.creative, adResponse.prebidKeywords, adResponse.refresh,
      adResponse.handoffUrls, adResponse.selectedUrls, adResponse.impressionUrls, adResponse.clickUrls,
      adResponse.expireUrls, adResponse.lossUrls, adResponse.errorUrls, adResponse.reserved,
      adResponse.expirationTimeMs, Winner.from(adResponse.winnerResponse), AdUnit.from(adResponse.adUnitResponse));
  }

  @VisibleForTesting
  public Ad(@NonNull ApiClient apiClient, @NonNull String adUnitId, @Nullable String creative,
            @Nullable String prebidKeywords, @Nullable Integer refreshTimeSeconds, @Nullable List<String> handoffUrls,
            @Nullable List<String> selectedUrls, @Nullable List<String> impressionUrls,
            @Nullable List<String> clickUrls, @Nullable List<String> expireUrls,
            @Nullable List<String> lossUrls, @Nullable List<String> errorUrls,
            @Nullable Boolean reserved, @Nullable Integer expirationTimeMs, @NonNull Winner winner,
            @NonNull AdUnit adUnit) {
    mApiClient = apiClient;
    mAdUnitId = adUnitId;
    mCreative = creative;
    mPrebidKeywords = prebidKeywords == null ? "" : prebidKeywords;
    mRefreshTimeSeconds = refreshTimeSeconds == null ? RefreshTimer.DEFAULT_REFRESH_TIME_SECONDS : refreshTimeSeconds;
    mHandoffUrls = handoffUrls == null ? Collections.<String>emptyList() : handoffUrls;
    mSelectedUrls = selectedUrls == null ? Collections.<String>emptyList() : selectedUrls;
    mImpressionUrls = impressionUrls == null ? Collections.<String>emptyList() : impressionUrls;
    mClickUrls = clickUrls == null ? Collections.<String>emptyList() : clickUrls;
    mExpireUrls = expireUrls == null ? Collections.<String>emptyList() : expireUrls;
    mLossUrls = lossUrls == null ? Collections.<String>emptyList() : lossUrls;
    mErrorUrls = errorUrls == null ? Collections.<String>emptyList() : errorUrls;
    mReserved = reserved == null ? false : reserved;
    mExpirationTimeMs = expirationTimeMs == null ? DEFAULT_EXPIRATION_MS : expirationTimeMs.longValue();
    mCreatedAtMs = System.currentTimeMillis();
    mWinner = winner;
    mAdUnit = adUnit;
    mSystemTimeHelper = new Helpers.SystemTimeHelper();
  }

  @NonNull
  public String getAdUnitId() {
    return mAdUnitId;
  }

  @Nullable
  public String getCreative() {
    return mCreative;
  }

  @NonNull
  public String getPrebidKeywords() {
    return mPrebidKeywords;
  }

  @NonNull
  public Integer getRefreshTimeSeconds() {
    return mRefreshTimeSeconds;
  }

  @NonNull
  public List<String> getHandoffUrls() {
    return mHandoffUrls;
  }

  @NonNull
  public List<String> getSelectedUrls() {
    return mSelectedUrls;
  }

  @NonNull
  public List<String> getImpressionUrls() {
    return mImpressionUrls;
  }

  @NonNull
  public List<String> getClickUrls() {
    return mClickUrls;
  }

  @NonNull
  public List<String> getExpireUrls() {
    return mExpireUrls;
  }

  @NonNull
  public List<String> getLossUrls() {
    return mLossUrls;
  }

  @NonNull
  public List<String> getErrorUrls() {
    return mErrorUrls;
  }

  public boolean isReserved() {
    return mReserved;
  }

  @NonNull
  public Winner getWinner() {
    return mWinner;
  }

  @NonNull
  public AdUnit getAdUnit() {
    return mAdUnit;
  }

  public long getExpirationTimeMs() {
    return mExpirationTimeMs;
  }

  public boolean isExpired() {
    return isExpired(mSystemTimeHelper);
  }

  @VisibleForTesting
  boolean isExpired(@NonNull Helpers.SystemTimeHelper systemTimeHelper) {
    return systemTimeHelper.currentTimeMillis() - mCreatedAtMs >= mExpirationTimeMs;
  }

  public void trackHandoffUrls() {
    if (mHandoffTracked) {
      return;
    }

    trackUrls(mHandoffUrls, TrackingType.HANDOFF);
    mHandoffTracked = true;
  }

  public void trackSelectedUrls() {
    if (mSelectedTracked) {
      return;
    }

    trackUrls(mSelectedUrls, TrackingType.SELECTED);
    mSelectedTracked = true;
  }

  public void trackImpressionUrls() {
    if (mImpressionTracked) {
      return;
    }

    trackUrls(mImpressionUrls, TrackingType.IMPRESSION);
    mImpressionTracked = true;
  }

  public void trackClickUrls() {
    if (mClickTracked) {
      return;
    }

    trackUrls(mClickUrls, TrackingType.CLICK);
    mClickTracked = true;
  }

  public void trackExpireUrls() {
    if (mExpireTracked) {
      return;
    }

    trackUrls(mExpireUrls, TrackingType.EXPIRE);
    mExpireTracked = true;
  }

  public void trackLossUrls() {
    trackLossUrls(LossUrlParameters.EMPTY_PARAMETERS);
  }

  public void trackLossUrls(@NonNull LossUrlParameters lossUrlParameters) {
    if (mLossTracked) {
      return;
    }

    final List<String> modifiedLossUrls = new ArrayList<>(mLossUrls.size());
    final Set<Map.Entry<String, String>> entries = lossUrlParameters.getMacroPairs().entrySet();
    for (String lossUrl : mLossUrls) {
      for (Map.Entry<String, String> entry : entries) {
        lossUrl = lossUrl.replace(entry.getKey(), entry.getValue());
      }
      modifiedLossUrls.add(lossUrl);
    }

    trackUrls(modifiedLossUrls, TrackingType.LOSS);
    mLossTracked = true;
  }

  public void trackError(@NonNull String errorMessage) {
    mApiClient.trackError(errorMessage);
  }

  private void trackUrls(@NonNull List<String> urls, @NonNull TrackingType type) {
    for (final String url : urls) {
      MaxAdsLog.d(TAG, "Tracking " + type.toString() + " for ad unit id: " + mAdUnitId + ". Url: " + url);
      mApiClient.trackUrl(url);
    }
  }

  public static class LossUrlParameters {
    @NonNull public static final LossUrlParameters EMPTY_PARAMETERS = new LossUrlParameters.Builder().build();
    @NonNull private static final String WINNING_PRICE_CENTS_MACRO = "${MAX_WINNING_PRICE_MACRO}";
    @NonNull private final Map<String, String> mMacroPairs;

    LossUrlParameters(@Nullable Integer winningPriceCents) {
      mMacroPairs = new HashMap<>(1);
      if (winningPriceCents != null) {
        mMacroPairs.put(WINNING_PRICE_CENTS_MACRO, String.valueOf(winningPriceCents));
      }
    }

    @NonNull
    public Map<String, String> getMacroPairs() {
      return mMacroPairs;
    }

    public static class Builder {
      @Nullable private Integer mWinningPriceCents;

      public Builder withWinningPriceCents(@Nullable Integer winningPriceCents) {
        mWinningPriceCents = winningPriceCents;
        return this;
      }

      public LossUrlParameters build() {
        return new LossUrlParameters(mWinningPriceCents);
      }
    }
  }
}
