package io.maxads.ads.base.cache;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import io.maxads.ads.base.model.Ad;
import io.maxads.ads.base.util.MaxAdsLog;

public class AdCache {
  @NonNull private static final String TAG = AdCache.class.getSimpleName();

  public static class Entry {
    private static final long EXPIRATION_TIME_MS = TimeUnit.HOURS.toMillis(1);
    @NonNull private final Ad mAd;
    private final long mCachedAtMs;

    public Entry(@NonNull Ad ad) {
      mAd = ad;
      mCachedAtMs = System.currentTimeMillis();
    }

    @NonNull
    public Ad getAd() {
      return mAd;
    }

    public boolean hasExpired() {
      return System.currentTimeMillis() - mCachedAtMs >= EXPIRATION_TIME_MS;
    }
  }

  @NonNull private final Map<String, Entry> mAdMap;

  public AdCache() {
    mAdMap = new HashMap<>();
  }

  public void put(@NonNull Ad ad) {
    doPut(ad.getAdUnitId(), new Entry(ad));
  }

  @VisibleForTesting
  void doPut(@NonNull String adUnitId, @NonNull Entry entry) {
    MaxAdsLog.d(TAG, "Caching ad for ad unit id: " + adUnitId);

    final Entry previousEntry = mAdMap.put(adUnitId, entry);
    if (previousEntry == null) {
      return;
    }

    MaxAdsLog.d(TAG, "Replacing previously cached ad for ad unit id: " + adUnitId);

    final Ad previousAd = previousEntry.getAd();
    if (previousEntry.hasExpired()) {
      previousAd.trackExpireUrls();
    } else {
      previousAd.trackLossUrls();
    }
  }

  @Nullable
  public Ad remove(@NonNull String adUnitId) {
    MaxAdsLog.d(TAG, "Fetching cached ad for ad unit id: " + adUnitId);
    final Entry entry = mAdMap.remove(adUnitId);
    if (entry == null) {
      MaxAdsLog.d(TAG, "Did not find cached ad for ad unit id: " + adUnitId);
      return null;
    }

    final Ad ad = entry.getAd();
    if (entry.hasExpired()) {
      MaxAdsLog.d(TAG, "Cached ad has expired for ad unit id: " + adUnitId);
      ad.trackExpireUrls();
      return null;
    }

    return ad;
  }
}
