package io.maxads.ads.base.api;

import android.annotation.SuppressLint;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import java.util.List;
import java.util.concurrent.TimeUnit;

import io.maxads.ads.base.Constants;
import io.maxads.ads.interstitial.vast3.VASTMacroProcessor;
import io.maxads.ads.interstitial.vast3.model.VASTTracker;
import io.maxads.ads.interstitial.vast3.xml_model.VASTXml;
import io.reactivex.Observable;
import io.reactivex.internal.functions.Functions;
import io.reactivex.schedulers.Schedulers;
import okhttp3.Interceptor;
import okhttp3.OkHttpClient;
import okhttp3.ResponseBody;
import retrofit2.Retrofit;
import retrofit2.adapter.rxjava2.RxJava2CallAdapterFactory;
import retrofit2.converter.simplexml.SimpleXmlConverterFactory;

public class VASTApiClient {
  @NonNull private final VASTApiService mVASTApiService;

  public VASTApiClient(@NonNull List<Interceptor> applicationInterceptors,
                       @NonNull List<Interceptor> networkInterceptors) {
    final OkHttpClient.Builder builder = new OkHttpClient.Builder();
    for (Interceptor applicationInterceptor : applicationInterceptors) {
      builder.addInterceptor(applicationInterceptor);
    }
    for (Interceptor networkInterceptor : networkInterceptors) {
      builder.addNetworkInterceptor(networkInterceptor);
    }

    final Retrofit retrofit = new Retrofit.Builder()
      .baseUrl(Constants.MAXADS_URL)
      .addConverterFactory(SimpleXmlConverterFactory.createNonStrict())
      .addCallAdapterFactory(RxJava2CallAdapterFactory.create())
      .client(builder.build())
      .build();

    mVASTApiService = retrofit.create(VASTApiService.class);
  }

  public Observable<VASTXml> getVastXml(@NonNull String url) {
    return mVASTApiService.getVastXml(url)
      .subscribeOn(Schedulers.io());
  }

  public Observable<ResponseBody> getMediaFile(@NonNull String url) {
    return mVASTApiService.getMediaFile(url)
      .subscribeOn(Schedulers.io());
  }

  @SuppressLint("CheckResult")
  public void fireVASTTracker(@NonNull VASTTracker vastTracker, @NonNull VASTMacroProcessor vastMacroProcessor) {
    if (vastTracker.isTracked() && !vastTracker.isRepeatable()) {
      return;
    }

    vastTracker.setTracked();

    final String url = vastMacroProcessor.processUrl(vastTracker.getUrl());
    mVASTApiService.trackUrl(url)
      .subscribeOn(Schedulers.io())
      .retryWhen(new ExponentialBackoff(url, Jitter.DEFAULT, 2, 30, TimeUnit.SECONDS, 6))
      .subscribe(Functions.emptyConsumer(), Functions.<Throwable>emptyConsumer());
  }

  public void fireVASTTrackers(@Nullable List<VASTTracker> vastTrackers,
                               @NonNull VASTMacroProcessor vastMacroProcessor) {
    if (vastTrackers == null) {
      return;
    }

    for (VASTTracker vastTracker : vastTrackers) {
      fireVASTTracker(vastTracker, vastMacroProcessor);
    }
  }
}

