package io.maxads.ads.base.api;

import android.annotation.SuppressLint;
import android.support.annotation.NonNull;

import java.util.List;
import java.util.concurrent.TimeUnit;

import io.maxads.ads.base.Constants;
import io.maxads.ads.base.model.Ad;
import io.reactivex.Observable;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.functions.Consumer;
import io.reactivex.functions.Function;
import io.reactivex.internal.functions.Functions;
import io.reactivex.schedulers.Schedulers;
import okhttp3.Interceptor;
import okhttp3.OkHttpClient;
import retrofit2.Retrofit;
import retrofit2.adapter.rxjava2.RxJava2CallAdapterFactory;
import retrofit2.converter.gson.GsonConverterFactory;

public class ApiClient {
  @NonNull private final ApiService mApiService;

  public ApiClient(@NonNull List<Interceptor> applicationInterceptors, @NonNull List<Interceptor> networkInterceptors) {
    final OkHttpClient.Builder builder = new OkHttpClient.Builder();
    for (Interceptor applicationInterceptor : applicationInterceptors) {
      builder.addInterceptor(applicationInterceptor);
    }
    for (Interceptor networkInterceptor : networkInterceptors) {
      builder.addNetworkInterceptor(networkInterceptor);
    }

    final Retrofit retrofit = new Retrofit.Builder()
      .baseUrl(Constants.MAXADS_URL)
      .addConverterFactory(GsonConverterFactory.create())
      .addCallAdapterFactory(RxJava2CallAdapterFactory.create())
      .client(builder.build())
      .build();
    mApiService = retrofit.create(ApiService.class);
  }

  public Observable<Ad> getAd(@NonNull final AdRequest adRequest) {
    return mApiService.getAd(adRequest.getAdUnitId(), adRequest)
      .subscribeOn(Schedulers.io())
      .retryWhen(new ExponentialBackoff(Jitter.DEFAULT, 2, 30, TimeUnit.SECONDS, 3))
      .observeOn(AndroidSchedulers.mainThread())
      .map(new Function<AdResponse, Ad>() {
        @Override
        public Ad apply(AdResponse adResponse) {
          return Ad.from(adRequest.getAdUnitId(), adResponse);
        }
      });
  }

  @SuppressLint("CheckResult")
  public void trackUrl(@NonNull String url) {
    mApiService.trackUrl(url)
      .subscribeOn(Schedulers.io())
      .retryWhen(new ExponentialBackoff(url, Jitter.DEFAULT, 2, 30, TimeUnit.SECONDS, 6))
      .subscribe(Functions.emptyConsumer(), Functions.<Throwable>emptyConsumer());
  }

  @SuppressLint("CheckResult")
  public void trackError(@NonNull String message) {
    new ErrorRequestFactory()
      .createErrorRequest(message)
      .subscribe(new Consumer<ErrorRequest>() {
        @Override
        public void accept(ErrorRequest errorRequest) {
          mApiService.trackError(errorRequest)
            .subscribeOn(Schedulers.io())
            .retryWhen(new ExponentialBackoff(Jitter.DEFAULT, 2, 30, TimeUnit.SECONDS, 6))
            .subscribe(Functions.emptyConsumer(), Functions.<Throwable>emptyConsumer());
        }
      });
  }
}
