package io.maxads.ads.base.api;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.google.gson.annotations.Expose;
import com.google.gson.annotations.SerializedName;

import java.util.Map;

import io.maxads.ads.base.Session;

public class AdRequest {
  // Transient means don't serialize / send this over the wire
  @NonNull private transient final String mAdUnitId;

  @SerializedName("v")
  @Expose
  @NonNull private final String mVersion;

  @SerializedName("sdk_v")
  @Expose
  @NonNull private final String mSdkVersion;

  @SerializedName("app_v")
  @Expose
  @NonNull private final String mAppVersion;

  @SerializedName("ifa")
  @Expose
  @NonNull private final String mIFA;

  @SerializedName("lmt")
  @Expose
  @NonNull private final Boolean mLMT;

  @SerializedName("vendor_id")
  @Expose
  @NonNull private final String mVendorId;

  @SerializedName("tz")
  @Expose
  @NonNull private final String mTimeZone;

  @SerializedName("locale")
  @Expose
  @NonNull private final String mLocale;

  @SerializedName("orientation")
  @Expose
  @NonNull private final String mOrientation;

  @SerializedName("w")
  @Expose
  @NonNull private final Integer mWidthPx;

  @SerializedName("h")
  @Expose
  @NonNull private final Integer mHeightPx;

  @SerializedName("browser_agent")
  @Expose
  @NonNull private final String mBrowserAgent;

  @SerializedName("model")
  @Expose
  @NonNull private final String mModel;

  @SerializedName("connectivity")
  @Expose
  @NonNull private final String mConnectivity;

  @SerializedName("carrier")
  @Expose
  @NonNull private final String mCarrierName;

  @SerializedName("mccmnc")
  @Expose
  @NonNull private final String mMccmnc;

  /**
   * Mapping of ad unit id to SSP and MAX session depth score
   */
  @SerializedName("session")
  @Expose
  @NonNull private final SessionRequest mSessionRequest;

  /**
   * Sum of all session scores
   */
  @SerializedName("session_depth")
  @Expose
  @NonNull private final Integer mSessionDepth;

  @SerializedName("tokens")
  @Expose
  @NonNull private final Map<String, String> mTokens;

  @SerializedName("location_tracking")
  @Expose
  @NonNull private final String mLocationTracking;

  @SerializedName("location")
  @Expose
  @NonNull private final LocationRequest mLocationRequest;

  @SerializedName("price_floor")
  @Expose
  @Nullable private final Integer mPriceFloorCents;

  @SerializedName("test")
  @Expose
  @Nullable private final Boolean mTest;

  private AdRequest(@NonNull String adUnitId, @NonNull String version, @NonNull String sdkVersion,
                    @NonNull String appVersion, @NonNull String ifa, @NonNull Boolean lmt,
                    @NonNull String vendorId, @NonNull String timeZone, @NonNull String locale,
                    @NonNull String orientation, @NonNull Integer widthPx, @NonNull Integer heightPx,
                    @NonNull String browserAgent, @NonNull String model, @NonNull String connectivity,
                    @NonNull String carrierName, @NonNull String mccmnc, @NonNull Session session,
                    @NonNull Map<String, String> tokens, @NonNull String locationTracking, @Nullable Double latitude,
                    @Nullable Double longitude, @Nullable String locationTimestamp, @Nullable Double horizontalAccuracy,
                    @Nullable Double verticalAccuracy, @Nullable Integer priceFloorCents, @Nullable Boolean test) {
    mAdUnitId = adUnitId;
    mVersion = version;
    mSdkVersion = sdkVersion;
    mAppVersion = appVersion;
    mIFA = ifa;
    mLMT = lmt;
    mVendorId = vendorId;
    mTimeZone = timeZone;
    mLocale = locale;
    mOrientation = orientation;
    mWidthPx = widthPx;
    mHeightPx = heightPx;
    mBrowserAgent = browserAgent;
    mModel = model;
    mConnectivity = connectivity;
    mCarrierName = carrierName;
    mMccmnc = mccmnc;
    mSessionRequest = new SessionRequest(session);
    mSessionDepth = session.getSumOfSessionScores();
    mTokens = tokens;
    mLocationTracking = locationTracking;
    mLocationRequest = new LocationRequest(latitude, longitude, locationTimestamp, horizontalAccuracy, verticalAccuracy);
    mPriceFloorCents = priceFloorCents;
    mTest = test;
  }

  @NonNull
  public String getAdUnitId() {
    return mAdUnitId;
  }

  public static class Builder {
    @NonNull private final String mAdUnitId;
    @NonNull private final String mVersion;
    @NonNull private final String mSdkVersion;
    @NonNull private final String mAppVersion;
    @NonNull private final String mIFA;
    @NonNull private final Boolean mLMT;
    @NonNull private final String mVendorId;
    @NonNull private final String mTimeZone;
    @NonNull private final String mLocale;
    @NonNull private final String mOrientation;
    @NonNull private final Integer mWidthPx;
    @NonNull private final Integer mHeightPx;
    @NonNull private final String mBrowserAgent;
    @NonNull private final String mModel;
    @NonNull private final String mConnectivity;
    @NonNull private final String mCarrierName;
    @NonNull private final String mMccmnc;
    @NonNull private final Session mSession;
    @NonNull private final Map<String, String> mTokens;
    @NonNull private final String mLocationTracking;
    @Nullable private Double mLatitude;
    @Nullable private Double mLongitude;
    @Nullable private String mLocationTimestamp;
    @Nullable private Double mHorizontalAccuracy;
    @Nullable private Double mVerticalAccuracy;
    @Nullable private Integer mPriceFloorCents;
    @Nullable private Boolean mTest;

    public Builder(@NonNull String adUnitId, @NonNull String version, @NonNull String sdkVersion,
                   @NonNull String appVersion, @NonNull String ifa, @NonNull Boolean lmt,
                   @NonNull String vendorId, @NonNull String timeZone, @NonNull String locale,
                   @NonNull String orientation, @NonNull Integer widthPx, @NonNull Integer heightPx,
                   @NonNull String browserAgent, @NonNull String model, @NonNull String connectivity,
                   @NonNull String carrierName, @NonNull String mccmnc, @NonNull Session session,
                   @NonNull Map<String, String> tokens, @NonNull String locationTracking) {
      mAdUnitId = adUnitId;
      mVersion = version;
      mSdkVersion = sdkVersion;
      mAppVersion = appVersion;
      mIFA = ifa;
      mLMT = lmt;
      mVendorId = vendorId;
      mTimeZone = timeZone;
      mLocale = locale;
      mOrientation = orientation;
      mWidthPx = widthPx;
      mHeightPx = heightPx;
      mBrowserAgent = browserAgent;
      mModel = model;
      mConnectivity = connectivity;
      mCarrierName = carrierName;
      mMccmnc = mccmnc;
      mSession = session;
      mTokens = tokens;
      mLocationTracking = locationTracking;
    }

    public AdRequest.Builder withLatitude(@Nullable Double latitude) {
      mLatitude = latitude;
      return this;
    }

    public AdRequest.Builder withLongitude(@Nullable Double longitude) {
      mLongitude = longitude;
      return this;
    }

    public AdRequest.Builder withLocationTimestamp(@Nullable String locationTimestamp) {
      mLocationTimestamp = locationTimestamp;
      return this;
    }

    public AdRequest.Builder withHorizontalAccuracy(@Nullable Double horizontalAccuracy) {
      mHorizontalAccuracy = horizontalAccuracy;
      return this;
    }

    public AdRequest.Builder withVerticalAccuracy(@Nullable Double verticalAccuracy) {
      mVerticalAccuracy = verticalAccuracy;
      return this;
    }

    public AdRequest.Builder withPriceFloorCents(@Nullable Integer priceFloorCents) {
      mPriceFloorCents = priceFloorCents;
      return this;
    }

    public AdRequest.Builder withTest(@Nullable Boolean test) {
      mTest = test;
      return this;
    }

    public AdRequest build() {
      return new AdRequest(mAdUnitId, mVersion, mSdkVersion, mAppVersion, mIFA, mLMT, mVendorId,
        mTimeZone, mLocale, mOrientation, mWidthPx, mHeightPx, mBrowserAgent, mModel,
        mConnectivity, mCarrierName, mMccmnc, mSession, mTokens, mLocationTracking, mLatitude, mLongitude,
        mLocationTimestamp, mHorizontalAccuracy, mVerticalAccuracy, mPriceFloorCents, mTest);
    }
  }
}
