package io.maxads.ads.base;

import android.support.annotation.NonNull;
import android.support.annotation.VisibleForTesting;

import com.jenzz.appstate.AppState;

import java.util.concurrent.TimeUnit;

import io.maxads.ads.base.util.Helpers;

/**
 * Session depth starts at 0 and is incremented for each ad request attempt regardless of whether a response is received
 * or not. When the app is backgrounded for more than 30 seconds, the session depth counter starts over again.
 */
public class SessionDepthManager implements AppStateManager.Listener {
  private static final long SESSION_RESET_THRESHOLD_MS = TimeUnit.SECONDS.toMillis(30);
  @NonNull private final Helpers.SystemTimeHelper mSystemTimeHelper;
  @NonNull private Session mSession;
  private long mTimeBackgroundedMs;

  public SessionDepthManager() {
    this(new Session(), new Helpers.SystemTimeHelper());
  }

  @VisibleForTesting
  SessionDepthManager(@NonNull Session session, @NonNull Helpers.SystemTimeHelper systemTimeHelper) {
    mSystemTimeHelper = systemTimeHelper;
    mSession = session;
  }

  @NonNull
  public Session getSession() {
    return mSession;
  }

  public void incrementMaxSessionDepth(@NonNull String adUnitId) {
    mSession.incrementMaxSessionDepth(adUnitId);
  }

  public void incrementSSPSessionDepth(@NonNull String adUnitId) {
    mSession.incrementSSPSessionDepth(adUnitId);
  }

  @Override
  public void onAppStateChanged(AppState appState) {
    if (appState == AppState.BACKGROUND) {
      mTimeBackgroundedMs = mSystemTimeHelper.currentTimeMillis();
    } else if (appState == AppState.FOREGROUND) {
      if (mSystemTimeHelper.currentTimeMillis() - mTimeBackgroundedMs > SESSION_RESET_THRESHOLD_MS) {
        mSession = new Session();
      }
    }
  }
}
