package io.maxads.ads.base;

import android.annotation.SuppressLint;
import android.app.Application;
import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.VisibleForTesting;

import java.lang.reflect.Method;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import io.maxads.ads.banner.BannerFactory;
import io.maxads.ads.base.api.ApiClient;
import io.maxads.ads.base.api.TokenManager;
import io.maxads.ads.base.api.VASTApiClient;
import io.maxads.ads.base.cache.AdCache;
import io.maxads.ads.base.cache.EmptyDiskLruCache;
import io.maxads.ads.base.cache.MaxDiskLruCache;
import io.maxads.ads.base.cache.MaxDiskLruCacheImpl;
import io.maxads.ads.base.cache.WebViewCache;
import io.maxads.ads.base.location.MaxLocationManager;
import io.maxads.ads.base.util.Checks;
import io.maxads.ads.base.util.MaxAdsLog;
import io.maxads.ads.interstitial.InterstitialFactory;
import okhttp3.Interceptor;

public class MaxAds {
  @NonNull private static final String TAG = MaxAds.class.getSimpleName();

  @NonNull private static volatile ApiClient sApiClient;
  @NonNull private static volatile VASTApiClient sVASTApiClient;
  @SuppressLint("StaticFieldLeak")
  @NonNull private static volatile DeviceInfo sDeviceInfo;
  @NonNull private static final SessionDepthManager sSessionDepthManager = new SessionDepthManager();
  @SuppressLint("StaticFieldLeak")
  @NonNull private static volatile MaxLocationManager sLocationManager;
  @NonNull private static volatile AppStateManager sAppStateManager;
  @NonNull private static volatile MaxDiskLruCache sMaxDiskLruCache;
  @NonNull private static final WebViewCache sWebViewCache = new WebViewCache();
  @NonNull private static final AdCache sAdCache = new AdCache();
  @NonNull private static final ModuleInitializationStateManager sModuleInitializationStateManager
    = new ModuleInitializationStateManager();
  @NonNull private static final TokenManager sTokenManager = new TokenManager();
  @NonNull private static final Map<String, BannerFactory> sPartnerBannerFactories = new HashMap<>();
  @NonNull private static final Map<String, InterstitialFactory> sPartnerInterstitialFactories = new HashMap<>();
  private static volatile boolean sInitialized;

  // We do this one here since we need to be able to inform the user if the initialization method has not been called
  static {
    sModuleInitializationStateManager.registerModuleInitializationState(new MaxSDKInitializationState());
  }

  /**
   * This method must be called to initialize the SDK before request ads.
   */
  public static void initialize(@NonNull Application application, boolean enableLocationTracking) {
    initialize(application, Collections.<Interceptor>emptyList(), Collections.<Interceptor>emptyList(),
      enableLocationTracking);
  }

  /**
   * For testing and debugging purposes only.
   */
  @VisibleForTesting
  public static void initialize(@NonNull Application application,
                                @NonNull List<Interceptor> applicationInterceptors,
                                @NonNull List<Interceptor> networkInterceptors,
                                boolean enableLocationTracking) {
    if (sInitialized) {
      MaxAdsLog.d(TAG, "MAX SDK is already initialized");
      return;
    }

    // Double check locking: https://medium.com/exploring-code/digesting-singleton-design-pattern-in-java-5d434f4f322
    if (!sInitialized) {
      synchronized (MaxAds.class) {
        if (!sInitialized) {
          MaxAdsLog.d(TAG, "Initializing MAX SDK");

          sApiClient = new ApiClient(applicationInterceptors, networkInterceptors);
          sVASTApiClient = new VASTApiClient(applicationInterceptors, networkInterceptors);
          sDeviceInfo = new DeviceInfo(application.getApplicationContext());
          sLocationManager = new MaxLocationManager(application.getApplicationContext());
          if (enableLocationTracking) {
            sLocationManager.enableLocationTracking();
          }

          final MaxDiskLruCache maxDiskLruCache = MaxDiskLruCacheImpl.initialize(application);
          sMaxDiskLruCache = maxDiskLruCache != null ? maxDiskLruCache : new EmptyDiskLruCache();

          sAppStateManager = new AppStateManager(application);
          sAppStateManager.addListener(sSessionDepthManager);
          sAppStateManager.addListener(sLocationManager);
          sAppStateManager.addListener(sWebViewCache);

          initializePartners(application.getApplicationContext(), sModuleInitializationStateManager,
            sTokenManager, sPartnerBannerFactories, sPartnerInterstitialFactories);

          sInitialized = true;
          MaxAdsLog.d(TAG, "MAX SDK successfully initialized");
        }
      }
    }
  }

  private static void initializePartners(@NonNull Context context,
                                         @NonNull ModuleInitializationStateManager moduleInitializationStateManager,
                                         @NonNull TokenManager tokenManager,
                                         @NonNull Map<String, BannerFactory> partnerBannerFactories,
                                         @NonNull Map<String, InterstitialFactory> partnerInterstitialFactories) {
    final Map<String, String> partners = new HashMap<>(2);
    partners.put("max-facebook", "io.maxads.facebook.MaxFacebook");
    partners.put("max-adcolony", "io.maxads.adcolony.MaxAdColony");

    for (Map.Entry<String, String> entry : partners.entrySet()) {
      final Class clazz;
      try {
        clazz = Class.forName(entry.getValue());
      } catch (ClassNotFoundException ignore) {
        MaxAdsLog.d(TAG, "Did not find partner module: " + entry.getKey());
        continue;
      }

      Method method;
      try {
        method = clazz.getMethod("initialize", Context.class, ModuleInitializationStateManager.class,
          TokenManager.class, Map.class, Map.class);
      } catch (NoSuchMethodException ignore) {
        MaxAdsLog.e(TAG, "Failed to find initialize method on partner module: " + entry.getKey());
        continue;
      }

      try {
        method.invoke(null, context, moduleInitializationStateManager, tokenManager, partnerBannerFactories,
          partnerInterstitialFactories);
      } catch (Exception ignore) {
        MaxAdsLog.e(TAG, "Failed to invoke initialize method on partner module: " + entry.getKey());
      }
    }
  }

  @NonNull
  public static ApiClient getApiClient() {
    checkInitialization();
    return sApiClient;
  }

  @NonNull
  public static VASTApiClient getVASTApiClient() {
    checkInitialization();
    return sVASTApiClient;
  }

  @NonNull
  public static DeviceInfo getDeviceInfo() {
    checkInitialization();
    return sDeviceInfo;
  }

  @NonNull
  public static SessionDepthManager getSessionDepthManager() {
    checkInitialization();
    return sSessionDepthManager;
  }

  @NonNull
  public static MaxLocationManager getLocationManager() {
    checkInitialization();
    return sLocationManager;
  }

  @NonNull
  public static AppStateManager getAppStateManager() {
    checkInitialization();
    return sAppStateManager;
  }

  @NonNull
  public static MaxDiskLruCache getMaxDiskLruCache() {
    checkInitialization();
    return sMaxDiskLruCache;
  }

  @NonNull
  public static WebViewCache getWebViewCache() {
    checkInitialization();
    return sWebViewCache;
  }

  @NonNull
  public static AdCache getAdCache() {
    checkInitialization();
    return sAdCache;
  }

  public static boolean areModulesInitialized() {
    return sModuleInitializationStateManager.areModulesInitialized();
  }

  @NonNull
  public static TokenManager getTokenManager() {
    checkInitialization();
    return sTokenManager;
  }

  @NonNull
  public static Map<String, BannerFactory> getPartnerBannerFactories() {
    checkInitialization();
    return sPartnerBannerFactories;
  }

  @NonNull
  public static Map<String, InterstitialFactory> getPartnerInterstitialFactories() {
    checkInitialization();
    return sPartnerInterstitialFactories;
  }

  public static boolean isInitialized() {
    return sInitialized;
  }

  private static void checkInitialization() {
    Checks.NoThrow.checkArgument(sInitialized, "MAX SDK has not been initialized. " +
      "Please call MaxAds#initialize before creating or invoking other maxads classes.");
  }
}
