package io.maxads.ads.interstitial.vast3.view;

import android.content.Context;
import android.graphics.Bitmap;
import android.media.MediaPlayer;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.view.View;
import android.widget.ImageView;
import android.widget.VideoView;

import io.maxads.ads.base.util.MaxAdsLog;

public class VASTViewModuleImpl implements VASTViewModule, MediaPlayer.OnPreparedListener,
  MediaPlayer.OnCompletionListener, MediaPlayer.OnErrorListener, View.OnClickListener {
  @NonNull private static final String TAG = VASTViewModuleImpl.class.getSimpleName();

  @NonNull private final VideoView mVideoView;
  @NonNull private final ImageView mLastVideoFrame;
  @NonNull private final ImageView mCountdownTimer;
  @NonNull private final ImageView mCloseButton;
  @NonNull private final VASTCountdownTimerDrawable mCountdownDrawable;

  @Nullable private VASTViewModule.Listener mListener;

  public VASTViewModuleImpl(@NonNull Context context,
                            @NonNull VideoView videoView,
                            @NonNull ImageView lastVideoFrame,
                            @NonNull ImageView countdownTimer,
                            @NonNull ImageView closeButton) {
    mVideoView = videoView;
    mLastVideoFrame = lastVideoFrame;
    mCountdownTimer = countdownTimer;
    mCloseButton = closeButton;
    mCountdownDrawable = new VASTCountdownTimerDrawable(context);

    mVideoView.setOnClickListener(this);
    mVideoView.setClickable(false);
    mVideoView.setOnPreparedListener(this);
    mVideoView.setOnCompletionListener(this);
    mVideoView.setOnErrorListener(this);
    mVideoView.requestFocus();

    mCountdownTimer.setImageDrawable(mCountdownDrawable);

    mCloseButton.setOnClickListener(this);
    mCloseButton.setImageDrawable(new VASTCloseButtonDrawable(context));
  }

  @Override
  public void setListener(@Nullable Listener listener) {
    mListener = listener;
  }

  @Override
  public void showCountdownTimer() {
    mCountdownTimer.setVisibility(View.VISIBLE);
  }

  @Override
  public void hideCountdownTimer() {
    mCountdownTimer.setVisibility(View.GONE);
  }

  @Override
  public void updateCountdownTimer(int totalTimeMs, int progressMs) {
    mCountdownDrawable.updateTimeRemaining(totalTimeMs, progressMs);
  }

  @Override
  public void showCloseButton() {
    mCloseButton.setVisibility(View.VISIBLE);
  }

  @Override
  public void setVideoPath(@NonNull String url) {
    try {
      mVideoView.setVideoPath(url);
    } catch (Exception e) {
      MaxAdsLog.e(TAG, e.getMessage(), e);
    }
  }

  @Override
  public void setVideoClickable(boolean clickable) {
    mVideoView.setClickable(clickable);
  }

  @Override
  public void startVideo() {
    mVideoView.start();
  }

  @Override
  public void stopVideo() {
    mVideoView.stopPlayback();
  }

  @Override
  public void videoSeekTo(int positionMs) {
    mVideoView.seekTo(positionMs);
  }

  @Override
  public boolean videoIsPlaying() {
    return mVideoView.isPlaying();
  }

  @Override
  public int videoGetCurrentPositionMs() {
    return mVideoView.getCurrentPosition();
  }

  @Override
  public int videoGetDurationMs() {
    return mVideoView.getDuration();
  }

  @Override
  public void showLastVideoFrame(@NonNull Bitmap bitmap) {
    mLastVideoFrame.setImageBitmap(bitmap);
    mLastVideoFrame.setVisibility(View.VISIBLE);
  }

  @Override
  public void onPrepared(MediaPlayer mp) {
    if (mListener != null) {
      mListener.onMediaPlayerPrepared(mp);
    }
  }

  @Override
  public void onCompletion(MediaPlayer mp) {
    if (mListener != null) {
      mListener.onMediaPlayerCompletion(mp);
    }
  }

  @Override
  public boolean onError(MediaPlayer mp, int what, int extra) {
    if (mListener != null) {
      mListener.onMediaPlayerError(mp, what, extra);
    }
    return true;
  }

  @Override
  public void onClick(View v) {
    if (mListener == null) {
      return;
    }

    if (v == mVideoView) {
      mListener.onVideoClicked();
    } else if (v == mCloseButton) {
      mListener.onCloseButtonClicked();
    }
  }
}
