package io.maxads.ads.interstitial.vast3.model;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;
import android.text.TextUtils;

import java.io.Serializable;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;

import io.maxads.ads.base.util.MaxAdsLog;
import io.maxads.ads.interstitial.vast3.VASTUtil;

public class VASTOffset implements Serializable {
  private static final long serialVersionUID = 0L;
  @NonNull private static final String TAG = VASTOffset.class.getSimpleName();

  public enum Type {
    ABSOLUTE,
    FRACTION,
  }

  private static final Pattern offsetAbsolute = Pattern.compile("\\d{2}:\\d{2}:\\d{2}(.\\d{3})?");
  private static final Pattern offsetPercentage = Pattern.compile("((\\d{1,2})|(100))%");

  @Nullable
  public static VASTOffset from(@Nullable String offset) {
    if (TextUtils.isEmpty(offset)) {
      return null;
    }

    final String offsetTrim = offset.trim();
    final Type type;
    Integer offsetMs = null;
    Float offsetFraction = null;

    if (offsetAbsolute.matcher(offsetTrim).matches()) {
      type = Type.ABSOLUTE;
      offsetMs = parseAbsoluteOffset(offset);
    } else if (offsetPercentage.matcher(offsetTrim).matches()) {
      type = Type.FRACTION;
      offsetFraction = parseFractionalOffset(offset);
    } else {
      MaxAdsLog.d(TAG, "Unable to create VAST offset from: " + offset);
      return null;
    }

    return new VASTOffset(offsetTrim, type, offsetMs, offsetFraction);
  }

  @NonNull
  public static VASTOffset from(int offsetMs) {
    offsetMs = offsetMs < 0 ? 0 : offsetMs;
    return new VASTOffset(VASTUtil.formatTimeMs(offsetMs), Type.ABSOLUTE, offsetMs, null);
  }

  @VisibleForTesting
  static int parseAbsoluteOffset(@NonNull String offset) {
    final String[] HHMMSSmmm = offset.split(":");
    if (HHMMSSmmm.length != 3) {
      return 0;
    }

    final long totalMs;
    try {
      totalMs = TimeUnit.HOURS.toMillis(Integer.parseInt(HHMMSSmmm[0]))
        + TimeUnit.MINUTES.toMillis(Integer.parseInt(HHMMSSmmm[1]))
        + (int)(Float.parseFloat(HHMMSSmmm[2]) * 1000);
    } catch (NumberFormatException e) {
      MaxAdsLog.w(TAG, "Unable to parse offset absolute value", e);
      return 0;
    }

    return (int) totalMs;
  }

  @VisibleForTesting
  static float parseFractionalOffset(@NonNull String offset) {
    try {
      return Float.parseFloat(offset.replace("%", "")) / 100f;
    } catch (NumberFormatException e) {
      MaxAdsLog.w(TAG, "Unable to parse offset fraction value", e);
      return 0;
    }
  }

  @NonNull private final String mOffset;
  @NonNull private final Type mType;
  @Nullable private final Integer mOffsetMs;
  @Nullable private final Float mOffsetFraction;

  public VASTOffset(@NonNull String offset,
                    @NonNull Type type,
                    @Nullable Integer offsetMs,
                    @Nullable Float offsetFraction) {
    mOffset = offset;
    mType = type;
    mOffsetMs = offsetMs;
    mOffsetFraction = offsetFraction;
  }

  @NonNull
  public String getOffset() {
    return mOffset;
  }

  @NonNull
  public Type getType() {
    return mType;
  }

  @Nullable
  public Integer getOffsetMs() {
    return mOffsetMs;
  }

  @Nullable
  public Float getOffsetFraction() {
    return mOffsetFraction;
  }

  public int getOffsetForVideoMs(int videoDurationMs) {
    int offsetMs = 0;
    switch (mType) {
      case ABSOLUTE:
        offsetMs = mOffsetMs != null ? mOffsetMs : 0;
        break;
      case FRACTION:
        offsetMs = mOffsetFraction != null ? Math.round(videoDurationMs * mOffsetFraction) : 0;
        break;
    }

    if (offsetMs > videoDurationMs) {
      offsetMs = videoDurationMs;
    }

    return offsetMs;
  }
}
