package io.maxads.ads.interstitial.vast3.model;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import io.maxads.ads.base.util.MaxAdsLog;
import io.maxads.ads.base.util.Util;
import io.maxads.ads.interstitial.vast3.xml_model.IconClicksXml;
import io.maxads.ads.interstitial.vast3.xml_model.IconXml;

public class VASTIconConfig implements Serializable {
  private static final long serialVersionUID = 0L;
  @NonNull private static final String TAG = VASTIconConfig.class.getSimpleName();

  private final int mWidthDp;
  private final int mHeightDp;
  private final int mWidthPx;
  private final int mHeightPx;
  @NonNull private final VASTResource mVASTResource;
  @NonNull private final List<VASTTracker> mCreativeViewTrackers;
  @Nullable private final String mClickThroughUrl;
  @NonNull private final List<VASTTracker> mClickTrackers;

  @Nullable
  public static VASTIconConfig from(@NonNull Context context, @Nullable IconXml iconXml) {
    if (iconXml == null) {
      return null;
    }

    final Integer widthDp = iconXml.width;
    if (widthDp == null || widthDp < 0 || widthDp > 300) {
      MaxAdsLog.d(TAG, "Invalid icon width: " + widthDp);
      return null;
    }

    final Integer heightDp = iconXml.height;
    if (heightDp == null || heightDp < 0 || heightDp > 300) {
      MaxAdsLog.d(TAG, "Invalid icon height: " + heightDp);
      return null;
    }

    final VASTResource vastResource = VASTResource.from(iconXml);
    if (vastResource == null) {
      return null;
    }

    final List<VASTTracker> creativeViewTrackers =
      VASTEventTracker.from(iconXml.iconViewTracking, VASTEventTracker.Event.creativeView);

    final IconClicksXml iconClicksXml = iconXml.iconClicksXml;
    String clickThroughUrl = null;
    List<VASTTracker> clickTrackers = new ArrayList<>();
    if (iconClicksXml != null) {
      clickThroughUrl = iconClicksXml.iconClickThrough;
      clickTrackers = VASTEventTracker.from(iconClicksXml.iconClickTracking, VASTEventTracker.Event.click);
    }

    return new VASTIconConfig(context, widthDp, heightDp, vastResource, creativeViewTrackers, clickThroughUrl,
      clickTrackers);
  }

  @VisibleForTesting
  VASTIconConfig(@NonNull Context context,
                 int widthDp,
                 int heightDp,
                 @NonNull VASTResource vastResource,
                 @NonNull List<VASTTracker> creativeViewTrackers,
                 @Nullable String clickThroughUrl,
                 @NonNull List<VASTTracker> clickTrackers) {
    mWidthDp = widthDp;
    mHeightDp = heightDp;
    mWidthPx = Util.dipsToIntPixels(widthDp, context);
    mHeightPx = Util.dipsToIntPixels(heightDp, context);
    mVASTResource = vastResource;
    mCreativeViewTrackers = creativeViewTrackers;
    mClickThroughUrl = clickThroughUrl != null ? clickThroughUrl.trim() : null;
    mClickTrackers = clickTrackers;
  }

  public int getWidthDp() {
    return mWidthDp;
  }

  public int getHeightDp() {
    return mHeightDp;
  }

  public int getWidthPx() {
    return mWidthPx;
  }

  public int getHeightPx() {
    return mHeightPx;
  }

  @NonNull
  public VASTResource getVASTResource() {
    return mVASTResource;
  }

  @NonNull
  public List<VASTTracker> getCreativeViewTrackers() {
    return mCreativeViewTrackers;
  }

  @Nullable
  public String getClickThroughUrl() {
    return mClickThroughUrl;
  }

  @NonNull
  public List<VASTTracker> getClickTrackers() {
    return mClickTrackers;
  }

  @Nullable
  public String getHtmlPayload() {
    return mVASTResource.getHtmlPayload(mWidthDp, mHeightDp);
  }
}
