package io.maxads.ads.interstitial.vast3.model;

import android.content.Context;
import android.graphics.Point;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;

import java.io.Serializable;
import java.util.List;

import io.maxads.ads.base.util.MaxAdsLog;
import io.maxads.ads.interstitial.vast3.xml_model.CompanionAdXml;

public class VASTCompanionAdConfig implements Serializable {
  private static final long serialVersionUID = 0L;
  @NonNull private static final String TAG = VASTCompanionAdConfig.class.getSimpleName();

  private final int mWidthDp;
  private final int mHeightDp;
  private final int mScaledWidthPx;
  private final int mScaledHeightPx;
  @NonNull private final VASTResource mVASTResource;
  @NonNull private final List<VASTTracker> mCreativeViewTrackers;
  @Nullable private final String mClickThroughUrl;
  @NonNull private final List<VASTTracker> mClickTrackers;

  @Nullable
  public static VASTCompanionAdConfig from(@NonNull Context context, @Nullable CompanionAdXml companionAdXml) {
    if (companionAdXml == null) {
      return null;
    }

    final Integer widthDp = companionAdXml.width;
    if (widthDp == null || widthDp < 0) {
      MaxAdsLog.d(TAG, "Invalid companion ad width: " + widthDp);
      return null;
    }

    final Integer heightDp = companionAdXml.height;
    if (heightDp == null || heightDp < 0) {
      MaxAdsLog.d(TAG, "Invalid companion ad height: " + heightDp);
      return null;
    }

    final VASTResource vastResource = VASTResource.from(companionAdXml);
    if (vastResource == null) {
      return null;
    }

    final Point scaledDimensionsPx = vastResource.getScaledDimensionsPx(context, widthDp, heightDp);
    final int scaledWidthPx = scaledDimensionsPx.x;
    final int scaledHeightPx = scaledDimensionsPx.y;

    final List<VASTTracker> creativeViewTrackers = VASTTracker.from(companionAdXml.trackingXmls);
    final String clickThroughUrl = companionAdXml.companionClickThrough;
    final List<VASTTracker> clickTrackers = VASTEventTracker.from(
      companionAdXml.companionClickTrackingUrls, VASTEventTracker.Event.click);

    return new VASTCompanionAdConfig(widthDp, heightDp, scaledWidthPx, scaledHeightPx, vastResource,
      creativeViewTrackers, clickThroughUrl, clickTrackers);
  }

  @VisibleForTesting
  VASTCompanionAdConfig(int widthDp,
                        int heightDp,
                        int scaledWidthPx,
                        int scaledHeightPx,
                        @NonNull VASTResource vastResource,
                        @NonNull List<VASTTracker> creativeViewTrackers,
                        @Nullable String clickThroughUrl,
                        @NonNull List<VASTTracker> clickTrackers) {
    mWidthDp = widthDp;
    mHeightDp = heightDp;
    mVASTResource = vastResource;
    mScaledWidthPx = scaledWidthPx;
    mScaledHeightPx = scaledHeightPx;
    mCreativeViewTrackers = creativeViewTrackers;
    mClickThroughUrl = clickThroughUrl != null ? clickThroughUrl.trim() : null;
    mClickTrackers = clickTrackers;
  }

  public int getWidthDp() {
    return mWidthDp;
  }

  public int getHeightDp() {
    return mHeightDp;
  }

  public int getScaledWidthPx() {
    return mScaledWidthPx;
  }

  public int getScaledHeightPx() {
    return mScaledHeightPx;
  }

  @NonNull
  public VASTResource getVASTResource() {
    return mVASTResource;
  }

  @NonNull
  public List<VASTTracker> getCreativeViewTrackers() {
    return mCreativeViewTrackers;
  }

  @Nullable
  public String getClickThroughUrl() {
    return mClickThroughUrl;
  }

  @NonNull
  public List<VASTTracker> getClickTrackers() {
    return mClickTrackers;
  }

  @Nullable
  public String getHtmlPayload() {
    return mVASTResource.getHtmlPayload(mWidthDp, mHeightDp);
  }
}
