package io.maxads.ads.interstitial.vast3;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;
import android.support.annotation.WorkerThread;
import android.text.TextUtils;

import org.simpleframework.xml.core.Persister;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Callable;

import io.maxads.ads.base.util.MaxAdsLog;
import io.maxads.ads.interstitial.vast3.model.VASTEventTracker;
import io.maxads.ads.interstitial.vast3.model.VASTTracker;
import io.maxads.ads.interstitial.vast3.model.VASTVideoConfig;
import io.maxads.ads.interstitial.vast3.xml_model.AdXml;
import io.maxads.ads.interstitial.vast3.xml_model.CompanionAdXml;
import io.maxads.ads.interstitial.vast3.xml_model.CreativeXml;
import io.maxads.ads.interstitial.vast3.xml_model.IconXml;
import io.maxads.ads.interstitial.vast3.xml_model.InLineXml;
import io.maxads.ads.interstitial.vast3.xml_model.LinearXml;
import io.maxads.ads.interstitial.vast3.xml_model.MediaFileXml;
import io.maxads.ads.interstitial.vast3.xml_model.VASTXml;
import io.maxads.ads.interstitial.vast3.xml_model.WrapperXml;
import io.reactivex.Observable;
import io.reactivex.ObservableSource;
import io.reactivex.schedulers.Schedulers;

public class VASTProcessor {
  @NonNull public static final String TAG = VASTProcessor.class.getSimpleName();
  private static final int MAX_VAST_WRAPPERS = 10;

  @NonNull private final VASTApiClientDecorator mVASTApiClientDecorator;
  @NonNull private final MediaFilePicker mMediaFilePicker;
  @NonNull private final CompanionAdPicker mCompanionAdPicker;
  @NonNull private final IconPicker mIconPicker;
  @NonNull private final Persister mPersister;
  private int mVastWrapperLevel;

  public VASTProcessor(@NonNull VASTApiClientDecorator vastApiClientDecorator,
                       @NonNull MediaFilePicker mediaFilePicker,
                       @NonNull CompanionAdPicker companionAdPicker,
                       @NonNull IconPicker iconPicker) {
    this(vastApiClientDecorator, mediaFilePicker, companionAdPicker, iconPicker, new Persister());
  }

  private VASTProcessor(@NonNull VASTApiClientDecorator vastApiClientDecorator,
                @NonNull MediaFilePicker mediaFilePicker,
                @NonNull CompanionAdPicker companionAdPicker,
                @NonNull IconPicker iconPicker,
                @NonNull Persister persister) {
    mVASTApiClientDecorator = vastApiClientDecorator;
    mMediaFilePicker = mediaFilePicker;
    mCompanionAdPicker = companionAdPicker;
    mIconPicker = iconPicker;
    mPersister = persister;
  }

  @NonNull
  public Observable<VASTVideoConfig> processXmlString(@NonNull final String xml) {
    return Observable.defer(new Callable<ObservableSource<VASTVideoConfig>>() {
      @Override
      public ObservableSource<VASTVideoConfig> call() {
        final VASTVideoConfig vastVideoConfig = new VASTVideoConfig();
        final boolean success = doProcessXmlString(xml, vastVideoConfig);
        return success
          ? Observable.just(vastVideoConfig)
          : Observable.<VASTVideoConfig>error(new Exception("Failed to process VAST XML."));
      }
    }).subscribeOn(Schedulers.io());
  }

  @WorkerThread
  @VisibleForTesting
  boolean doProcessXmlString(@NonNull final String xml, @NonNull VASTVideoConfig vastVideoConfig) {
    final ArrayList<VASTTracker> vastErrorTrackers = new ArrayList<>();

    VASTXml vastXml;
    try {
      vastXml = mPersister.read(VASTXml.class, xml);
    } catch (Exception e) {
      processVASTError(vastErrorTrackers, VASTError.XML_PARSING_ERROR);
      return false;
    }

    return processVASTXml(vastXml, vastVideoConfig, vastErrorTrackers);
  }

  @WorkerThread
  private boolean processVASTXml(@NonNull VASTXml vastXml,
                                 @NonNull VASTVideoConfig vastVideoConfig,
                                 @NonNull List<VASTTracker> vastErrorTrackers) {
    if (mVastWrapperLevel > MAX_VAST_WRAPPERS) {
      processVASTError(vastErrorTrackers, VASTError.WRAPPER_LIMIT_REACHED);
      return false;
    }

    if (vastXml.adXmls == null) {
      return false;
    }

    for (AdXml adXml : vastXml.adXmls) {
      if (processInLineXml(adXml.inLineXml, vastVideoConfig, new ArrayList<>(vastErrorTrackers))) {
        return true;
      } else if (processWrapperXml(adXml.wrapperXml, vastVideoConfig, new ArrayList<>(vastErrorTrackers))) {
        return true;
      }
    }

    return false;
  }

  @WorkerThread
  private boolean processInLineXml(@Nullable InLineXml inLineXml,
                                   @NonNull VASTVideoConfig vastVideoConfig,
                                   @NonNull List<VASTTracker> vastErrorTrackers) {
    if (inLineXml == null || inLineXml.creativeXmls == null) {
      return false;
    }

    vastErrorTrackers.addAll(VASTEventTracker.from(inLineXml.errorUrl, VASTEventTracker.Event.error));

    // Look for a valid media file
    boolean foundMediaFile = false;
    for (CreativeXml creativeXml : inLineXml.creativeXmls) {
      if (processLinearXml(creativeXml.linearXml, vastVideoConfig, vastErrorTrackers, true)) {
        vastVideoConfig.addImpressionTrackers(
          VASTEventTracker.from(inLineXml.impressionUrls, VASTEventTracker.Event.impression));
        foundMediaFile = true;
        break;
      }
    }

    // If we found a valid media file then search for the first valid companion ad
    if (foundMediaFile) {
      for (CreativeXml creativeXml : inLineXml.creativeXmls) {
        if (processCompanionAdsXml(creativeXml.companionAdXmls, vastVideoConfig)) {
          break;
        }
      }
    }

    return foundMediaFile;
  }

  @VisibleForTesting
  @WorkerThread
  boolean processWrapperXml(@Nullable WrapperXml wrapperXml,
                            @NonNull VASTVideoConfig vastVideoConfig,
                            @NonNull List<VASTTracker> vastErrorTrackers) {
    if (wrapperXml == null || TextUtils.isEmpty(wrapperXml.vastAdTagUri)) {
      return false;
    }

    vastErrorTrackers.addAll(VASTEventTracker.from(wrapperXml.errorUrl, VASTEventTracker.Event.error));

    final VASTXml vastXml;
    try {
      vastXml = mVASTApiClientDecorator.getVASTXml(wrapperXml.vastAdTagUri.trim()).blockingSingle();
    } catch (Exception e) {
      processVASTError(vastErrorTrackers, VASTError.WRAPPER_TIMEOUT);
      return false;
    }

    mVastWrapperLevel++;
    final boolean success = processVASTXml(vastXml, vastVideoConfig, vastErrorTrackers);
    mVastWrapperLevel--;

    // If we found a valid media file then aggregate impression, and event trackers
    // at the wrapper level
    if (success) {
      vastVideoConfig.addImpressionTrackers(
        VASTEventTracker.from(wrapperXml.impressionUrls, VASTEventTracker.Event.impression));
      if (wrapperXml.creativeXmls != null) {
        for (CreativeXml creativeXml : wrapperXml.creativeXmls) {
          // Aggregate trackers and look for an icon if don't already have one
          processLinearXml(creativeXml.linearXml, vastVideoConfig, vastErrorTrackers, false);

          // Look for a valid companion ad if we still haven't found one
          processCompanionAdsXml(creativeXml.companionAdXmls, vastVideoConfig);
        }
      }
    }

    return success;
  }

  @WorkerThread
  private boolean processLinearXml(@Nullable LinearXml linearXml,
                                   @NonNull VASTVideoConfig vastVideoConfig,
                                   @NonNull List<VASTTracker> vastErrorTrackers,
                                   boolean findMediaFile) {
    if (findMediaFile) {
      if (linearXml == null) {
        return false;
      }

      final MediaFileXml mediaFileXml = mMediaFilePicker.pickMediaFile(mVASTApiClientDecorator, vastErrorTrackers,
        linearXml.mediaFileXmls);
      if (mediaFileXml == null) {
        return false;
      }
      vastVideoConfig.setMediaFileUrl(mediaFileXml.value);
      vastVideoConfig.setClickThroughUrl(linearXml.videoClicksXml != null
        ? linearXml.videoClicksXml.clickThrough : null);

      // When we find a valid media file, we add all the error trackers we've been forwarding
      vastVideoConfig.addErrorTracker(vastErrorTrackers);
    }

    if (linearXml != null) {
      vastVideoConfig.addVASTTrackers(VASTTracker.from(linearXml.trackingXmls));
      if (linearXml.videoClicksXml != null) {
        vastVideoConfig.addClickTrackers(
          VASTEventTracker.from(linearXml.videoClicksXml.clickTracking, VASTEventTracker.Event.click));
      }

      processIconsXml(linearXml.iconXmls, vastVideoConfig);
    }

    return true;
  }

  @WorkerThread
  private boolean processCompanionAdsXml(@Nullable List<CompanionAdXml> companionAdXmls,
                                         @NonNull VASTVideoConfig vastVideoConfig) {
    if (vastVideoConfig.getVASTCompanionAdConfig() != null) {
      return true;
    }

    vastVideoConfig.setVASTCompanionAdConfig(mCompanionAdPicker.pickCompanionAd(companionAdXmls));

    return vastVideoConfig.getVASTCompanionAdConfig() != null;
  }


  @WorkerThread
  private boolean processIconsXml(@Nullable List<IconXml> iconXmls, @NonNull VASTVideoConfig vastVideoConfig) {
    if (vastVideoConfig.getVASTIconConfig() != null) {
      return true;
    }

    vastVideoConfig.setVASTIconConfig(mIconPicker.pickIcon(iconXmls));

    return vastVideoConfig.getVASTIconConfig() != null;
  }

  private void processVASTError(@NonNull List<VASTTracker> vastErrorTrackers, @NonNull VASTError vastError) {
    mVASTApiClientDecorator.fireVASTTrackers(vastErrorTrackers, VASTMacroDataImpl.from(vastError));
    MaxAdsLog.w(TAG, vastError.toString());
  }
}
