package io.maxads.ads.interstitial.vast3;

import android.annotation.SuppressLint;
import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;

import io.maxads.ads.base.MaxAds;
import io.maxads.ads.base.cache.MaxDiskLruCache;
import io.maxads.ads.base.util.MaxAdsLog;
import io.maxads.ads.interstitial.vast3.model.VASTVideoConfig;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.disposables.Disposable;
import io.reactivex.functions.Action;
import io.reactivex.functions.Consumer;
import io.reactivex.internal.functions.Functions;
import okhttp3.ResponseBody;

public class VASTPreparer {
  @NonNull public static final String TAG = VASTPreparer.class.getSimpleName();

  public interface Listener {
    void onPrepared(@NonNull VASTVideoConfig vastVideoConfig);
    void onError();
  }

  @NonNull private final VASTApiClientDecorator mVASTApiClientDecorator;
  @NonNull private final MaxDiskLruCache mMaxDiskLruCache;
  @NonNull private final VASTProcessor mVASTProcessor;
  private final int mSkipOffsetMs;
  private final int mMaxDurationMs;
  private boolean mDestroyed;

  @Nullable private Disposable mProcessXmlDisposable;
  @Nullable private Disposable mDownloadMediaFileDisposable;
  @Nullable private Disposable mCacheMediaFileDisposable;

  @Nullable private Listener mListener;

  public VASTPreparer(@NonNull Context context, int skipOffsetMs, int maxDurationMs) {
    this(new VASTApiClientDecorator(MaxAds.getVASTApiClient()),
      MaxAds.getMaxDiskLruCache(),
      new VASTProcessor(
        new VASTApiClientDecorator(MaxAds.getVASTApiClient()),
        new MediaFilePicker(context),
        new CompanionAdPicker(context),
        new IconPicker(context)),
      skipOffsetMs,
      maxDurationMs);
  }

  @VisibleForTesting
  VASTPreparer(@NonNull VASTApiClientDecorator vastApiClientDecorator,
               @NonNull MaxDiskLruCache maxDiskLruCache,
               @NonNull VASTProcessor VASTProcessor,
               int skipOffsetMs,
               int maxDurationMs) {
    mVASTApiClientDecorator = vastApiClientDecorator;
    mMaxDiskLruCache = maxDiskLruCache;
    mVASTProcessor = VASTProcessor;
    mSkipOffsetMs = skipOffsetMs;
    mMaxDurationMs = maxDurationMs;
  }

  public void setListener(@NonNull Listener listener) {
    mListener = listener;
  }

  public void destroy() {
    if (mProcessXmlDisposable != null) {
      mProcessXmlDisposable.dispose();
    }
    if (mDownloadMediaFileDisposable != null) {
      mDownloadMediaFileDisposable.dispose();
    }
    if (mCacheMediaFileDisposable != null) {
      mCacheMediaFileDisposable.dispose();
    }
    mDestroyed = true;
  }

  public void prepareVideo(@NonNull String xml) {
    if (mDestroyed) {
      return;
    }

    parseVASTXml(xml);
  }

  private void parseVASTXml(@NonNull String xml) {
    mProcessXmlDisposable = mVASTProcessor.processXmlString(xml)
      .observeOn(AndroidSchedulers.mainThread())
      .subscribe(
        new Consumer<VASTVideoConfig>() {
          @Override
          public void accept(VASTVideoConfig vastVideoConfig) {
            vastVideoConfig.setSkipOffset(mSkipOffsetMs);
            vastVideoConfig.setMaxDurationMs(mMaxDurationMs);
            downloadVASTMediaFile(vastVideoConfig);
          }
        },
        new Consumer<Throwable>() {
          @Override
          public void accept(Throwable throwable) {
            onError();
          }
        });
  }

  @SuppressLint("CheckResult")
  private void downloadVASTMediaFile(@NonNull final VASTVideoConfig vastVideoConfig) {
    final String mediaFileUrl = vastVideoConfig.getMediaFileUrl();
    if (mMaxDiskLruCache.containsKey(mediaFileUrl)) {
      onPrepared(vastVideoConfig);
      return;
    }

    mDownloadMediaFileDisposable = mVASTApiClientDecorator.getMediaFile(mediaFileUrl)
      .observeOn(AndroidSchedulers.mainThread())
      .subscribe(
        new Consumer<ResponseBody>() {
          @Override
          public void accept(ResponseBody responseBody) {
            cacheVASTMediaFile(vastVideoConfig, responseBody);
          }
        },
        new Consumer<Throwable>() {
          @Override
          public void accept(Throwable throwable) {
            final VASTError vastError = VASTError.MEDIA_FILE_NOT_FOUND;
            mVASTApiClientDecorator.fireVASTTrackers(vastVideoConfig.getErrorTrackers(),
              VASTMacroDataImpl.from(vastError));
            MaxAdsLog.w(TAG, vastError.toString());
            onError();
          }
        });
  }

  @SuppressLint("CheckResult")
  private void cacheVASTMediaFile(@NonNull final VASTVideoConfig vastVideoConfig, @NonNull ResponseBody responseBody) {
    mCacheMediaFileDisposable = mMaxDiskLruCache.putAsync(vastVideoConfig.getMediaFileUrl(), responseBody.source(), responseBody.contentLength())
      .observeOn(AndroidSchedulers.mainThread())
      .subscribe(
        Functions.emptyConsumer(),
        new Consumer<Throwable>() {
          @Override
          public void accept(Throwable throwable) {
            onError();
          }
        },
        new Action() {
          @Override
          public void run() {
            onPrepared(vastVideoConfig);
          }
        });
  }

  private void onPrepared(@NonNull VASTVideoConfig vastVideoConfig) {
    vastVideoConfig.setMediaFileDiskUrl(mMaxDiskLruCache.getFilePath(vastVideoConfig.getMediaFileUrl()));
    if (mListener != null) {
      mListener.onPrepared(vastVideoConfig);
    }
  }

  private void onError() {
    if (mListener != null) {
      mListener.onError();
    }
  }
}
