package io.maxads.ads.interstitial;

import android.app.Activity;
import android.content.Intent;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;

import io.maxads.ads.base.model.Ad;
import io.maxads.ads.base.util.MaxAdsLog;
import io.maxads.ads.interstitial.vast3.VASTPreparer;
import io.maxads.ads.interstitial.vast3.activity.VASTActivity;
import io.maxads.ads.interstitial.vast3.model.VASTVideoConfig;

public class VAST3Interstitial implements Interstitial, VASTPreparer.Listener, InterstitialBroadcastReceiver.Listener {
  @NonNull private static final String TAG = VAST3Interstitial.class.getSimpleName();

  public static class VAST3InterstitialFactory implements InterstitialFactory {
    @NonNull private final Activity mActivity;

    public VAST3InterstitialFactory(@NonNull Activity activity) {
      mActivity = activity;
    }

    @Nullable
    @Override
    public Interstitial createInterstitial(@NonNull Ad ad) {
      return new VAST3Interstitial(mActivity, ad);
    }
  }

  @NonNull private final Activity mActivity;
  @NonNull private final Ad mAd;
  @NonNull private final VASTPreparer mVASTPreparer;
  @NonNull private final InterstitialBroadcastReceiver mInterstitialBroadcastReceiver;

  @Nullable private VASTVideoConfig mVASTVideoConfig;
  @Nullable private Interstitial.Listener mListener;

  public VAST3Interstitial(@NonNull Activity activity, @NonNull Ad ad) {
    this(activity, ad, new VASTPreparer(activity, ad.getAdUnit().getSkipOffsetMs(), ad.getAdUnit().getMaxDurationMs()),
      new InterstitialBroadcastReceiver(activity));
  }

  VAST3Interstitial(@NonNull Activity activity,
                    @NonNull Ad ad,
                    @NonNull VASTPreparer vastPreparer,
                    @NonNull InterstitialBroadcastReceiver interstitialBroadcastReceiver) {
    mActivity = activity;
    mAd = ad;
    mVASTPreparer = vastPreparer;
    mVASTPreparer.setListener(this);
    mInterstitialBroadcastReceiver = interstitialBroadcastReceiver;
    mInterstitialBroadcastReceiver.setListener(this);
  }

  @Override
  public void setListener(@Nullable Listener listener) {
    mListener = listener;
  }

  @Override
  public void load() {
    mVASTPreparer.prepareVideo(mAd.getCreative());
  }

  @Override
  public void show() {
    doShow(mVASTVideoConfig);
  }

  @VisibleForTesting
  void doShow(@Nullable VASTVideoConfig vastVideoConfig) {
    if (vastVideoConfig == null) {
      MaxAdsLog.w(TAG, "VASTVideoConfig is null. Can't play video");
      return;
    }

    mInterstitialBroadcastReceiver.register();

    final Intent intent = new Intent(mActivity, VASTActivity.class);
    intent.putExtra(VASTActivity.VAST_VIDEO_CONFIG_KEY, vastVideoConfig);
    intent.putExtra(VASTActivity.BROADCAST_ID_KEY, mInterstitialBroadcastReceiver.getBroadcastId());
    intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
    mActivity.startActivity(intent);

    if (mListener != null) {
      mListener.onInterstitialShown(this);
    }
  }

  @Override
  public void destroy() {
    mVASTPreparer.destroy();
    mInterstitialBroadcastReceiver.destroy();
    mListener = null;
  }

  // VASTPreparer.Listener

  @Override
  public void onPrepared(@NonNull VASTVideoConfig vastVideoConfig) {
    mVASTVideoConfig = vastVideoConfig;
    if (mListener != null) {
      mListener.onInterstitialLoaded(this);
    }
  }

  @Override
  public void onError() {
    if (mListener != null) {
      mListener.onInterstitialFailedToLoad(this);
    }
  }

  // InterstitialBroadcastReceiver.Listener

  @Override
  public void onReceivedAction(@NonNull InterstitialBroadcastReceiver.Action action) {
    mInterstitialBroadcastReceiver.handleAction(action, this, mListener);
  }
}
