package io.maxads.ads.base.view;

import android.content.Context;
import android.graphics.Canvas;
import android.graphics.ColorFilter;
import android.graphics.Paint;
import android.graphics.PixelFormat;
import android.graphics.drawable.Drawable;
import android.support.annotation.ColorInt;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;

import io.maxads.ads.base.util.Util;

public class CloseButtonDrawable extends Drawable {
  private static final int X_WIDTH_DP = 2;
  private static final int OUTLINE_WIDTH_DP = 1;

  @NonNull private final Paint mXPaint;
  @NonNull private final Paint mOutlinePaint;

  public CloseButtonDrawable(@NonNull Context context,
                             @ColorInt int xColor,
                             @ColorInt int outlineColor,
                             @NonNull Paint.Style outlineStyle) {
    final float xStrokeWidthPx = Util.dipsToPixels(X_WIDTH_DP, context);
    float outlineStrokeWidthPx = Util.dipsToPixels(OUTLINE_WIDTH_DP, context);

    mXPaint = new Paint();
    mXPaint.setColor(xColor);
    mXPaint.setStyle(Paint.Style.STROKE);
    mXPaint.setStrokeWidth(xStrokeWidthPx);
    mXPaint.setStrokeCap(Paint.Cap.ROUND);
    mXPaint.setAntiAlias(true);

    mOutlinePaint = new Paint();
    mOutlinePaint.setColor(outlineColor);
    mOutlinePaint.setStyle(outlineStyle);
    mOutlinePaint.setStrokeWidth(outlineStrokeWidthPx);
    mOutlinePaint.setAntiAlias(true);
  }

  @Override
  public void draw(@NonNull Canvas canvas) {
    doDraw(canvas, getBounds().centerX(), getBounds().centerY(), getBounds().width(), getBounds().height());
  }

  @VisibleForTesting
  void doDraw(@NonNull Canvas canvas, int centerX, int centerY, int width, int height) {
    // Draw a circle centered inside the rectangle using the smallest length as radius
    final int radius = Math.min(centerX, centerY);
    canvas.drawCircle(centerX, centerY, radius, mOutlinePaint);

    // Draw an X centered inside the circle
    final float widthOffset = calculateWidthOffset(width, height);
    final float heightOffset = calculateHeightOffset(width, height);

    final int squareLength = Math.min(width, height);
    final float distanceToX = calculateDistanceToX(squareLength);

    // Bottom left to top right
    canvas.drawLine(widthOffset + distanceToX, (heightOffset + squareLength) - distanceToX,
      (widthOffset + squareLength) - distanceToX, heightOffset + distanceToX, mXPaint);

    // Top left to bottom right
    canvas.drawLine(widthOffset + distanceToX, heightOffset + distanceToX,
      (widthOffset + squareLength) - distanceToX, (heightOffset + squareLength) - distanceToX, mXPaint);
  }

  /**
   * For rectangles with w > h with the square centered in the rectangle; return the width offset to the square edge
   */
  @VisibleForTesting
  float calculateWidthOffset(int w, int h) {
    return w <= h ? 0 : (w - h) / 2;
  }

  /**
   * For rectangles with h > w with the square centered in the rectangle; return the height offset to the square edge
   */
  @VisibleForTesting
  float calculateHeightOffset(int w, int h) {
    return h <= w ? 0 : (h - w) / 2;
  }

  /**
   * Calculate the horizontal distance from the corner of the square to the start of the X
   * Since it's a square, this is also the vertical distance
   */
  @VisibleForTesting
  float calculateDistanceToX(int squareLength) {
    // c^2 = a^2 + b^2; (for square) c^2 = a^2 + a^2; c = sqrt(2a^2); c = a * sqrt(2)
    final double hypotenuseOfSquare = squareLength * Math.sqrt(2);

    // For circle inside of square, calculate the distance from the corner of the square to edge of circle
    final double cornerToCircle = 0.5 * (hypotenuseOfSquare - squareLength);

    // We want the X to be centered inside the circle and its edges to fill 52.5% of the diameter of the circle
    // This calculates the distance from the edge of the circle to the start of the X
    final double circleToX = (squareLength - (0.525 * squareLength)) / 2;

    // Calculate the distance from teh corner of the square to the start of the X
    final double cornerToX = cornerToCircle + circleToX;

    // We now have the hypotenuse of the equilateral triangle formed from the corner of the square to the start of the X
    // Calculate the length of the side of this triangle
    // a^2 + b^2 = c^2; a^2 + a^2 = c^2; 2a^2 = c^2; a^2 = c^2 / 2; a = sqrt(c^2 / 2); a = c / sqrt(2)
    final double distanceToX = cornerToX / Math.sqrt(2);
    return (float) distanceToX;
  }

  @Override
  public void setAlpha(int alpha) {
  }

  @Override
  public void setColorFilter(@Nullable ColorFilter colorFilter) {
  }

  @Override
  public int getOpacity() {
    return PixelFormat.UNKNOWN;
  }
}
