package io.maxads.ads.base.model;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import java.util.concurrent.TimeUnit;

import io.maxads.ads.base.Constants;
import io.maxads.ads.base.api.AdUnitResponse;

public class AdUnit {
  static final int UNDEFINED_DIMENSION = -1;

  public enum Format {
    BANNER,
    BANNER300X250,
    BANNER50,
    BANNER90,
    NATIVE,
    INSTREAM,
    INTERSTITIAL,
    EMPTY;

    @NonNull
    public static Format from(@Nullable String format) {
      if ("banner".equalsIgnoreCase(format)) {
        return BANNER;
      } else if ("banner300x250".equalsIgnoreCase(format)) {
        return BANNER300X250;
      } else if ("banner50".equalsIgnoreCase(format)) {
        return BANNER50;
      } else if ("banner90".equalsIgnoreCase(format)) {
        return BANNER90;
      } else if ("native".equalsIgnoreCase(format)) {
        return NATIVE;
      } else if ("instream".equalsIgnoreCase(format)) {
        return INSTREAM;
      } else if ("interstitial".equalsIgnoreCase(format)) {
        return INTERSTITIAL;
      } else {
        return EMPTY;
      }
    }
  }

  @NonNull private final Format mFormat;
  private final int mWidth;
  private final int mHeight;
  private final int mInterstitialLoadTimeoutMs;
  // VAST video max duration
  private final int mMaxDurationMs;
  // VAST video skip offset
  private final int mSkipOffsetMs;

  @NonNull
  public static AdUnit from(@Nullable AdUnitResponse adUnitResponse) {
    if (adUnitResponse == null) {
      return new AdUnit(Format.from(null), UNDEFINED_DIMENSION, UNDEFINED_DIMENSION, null, null, null);
    }

    return new AdUnit(Format.from(adUnitResponse.format), adUnitResponse.width, adUnitResponse.height,
      adUnitResponse.interstitialLoadTimeoutMs, adUnitResponse.skipOffsetSeconds, adUnitResponse.maxDurationSeconds);
  }

  public AdUnit(@NonNull Format format,
                int width,
                int height,
                @Nullable Integer interstitialLoadTimeoutMs,
                @Nullable Integer skipOffsetSeconds,
                @Nullable Integer maxDurationSeconds) {
    mFormat = format;
    mWidth = width;
    mHeight = height;
    mInterstitialLoadTimeoutMs = interstitialLoadTimeoutMs == null || interstitialLoadTimeoutMs < 0
      ? Constants.DEFAULT_INTERSTITIAL_LOAD_TIMEOUT_MS : interstitialLoadTimeoutMs;
    mSkipOffsetMs = skipOffsetSeconds == null || skipOffsetSeconds < 0
      ? Constants.DEFAULT_VAST_SKIP_OFFSET_MS : (int) TimeUnit.SECONDS.toMillis(skipOffsetSeconds);
    mMaxDurationMs = maxDurationSeconds == null || maxDurationSeconds < 0
      ? Constants.DEFAULT_VAST_MAX_DURATION_MS : (int) TimeUnit.SECONDS.toMillis(maxDurationSeconds);
  }

  @NonNull
  public Format getFormat() {
    return mFormat;
  }

  public int getWidth() {
    return mWidth;
  }

  public int getHeight() {
    return mHeight;
  }

  public int getInterstitialLoadTimeoutMs() {
    return mInterstitialLoadTimeoutMs;
  }

  public int getMaxDurationMs() {
    return mMaxDurationMs;
  }

  public int getSkipOffsetMs() {
    return mSkipOffsetMs;
  }
}
