package io.maxads.ads.base.api;

import android.location.Location;
import android.support.annotation.NonNull;
import android.support.annotation.VisibleForTesting;

import com.google.android.gms.ads.identifier.AdvertisingIdClient;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;
import java.util.Map;
import java.util.TimeZone;

import io.maxads.ads.base.Constants;
import io.maxads.ads.base.DeviceInfo;
import io.maxads.ads.base.MaxAds;
import io.maxads.ads.base.SessionDepthManager;
import io.maxads.ads.base.location.MaxLocationManager;
import io.maxads.ads.base.util.Optional;
import io.reactivex.Observable;
import io.reactivex.functions.Function3;

public class AdRequestFactory {
  @NonNull private final DeviceInfo mDeviceInfo;
  @NonNull private final SessionDepthManager mSessionDepthManager;
  @NonNull private final MaxLocationManager mLocationManager;
  @NonNull private final TokenManager mTokenManager;

  public AdRequestFactory() {
    this(MaxAds.getDeviceInfo(), MaxAds.getSessionDepthManager(), MaxAds.getLocationManager(), MaxAds.getTokenManager());
  }

  @VisibleForTesting
  AdRequestFactory(@NonNull DeviceInfo deviceInfo,
                   @NonNull SessionDepthManager sessionDepthManager,
                   @NonNull MaxLocationManager locationManager,
                   @NonNull TokenManager tokenManager) {
    mDeviceInfo = deviceInfo;
    mSessionDepthManager = sessionDepthManager;
    mLocationManager = locationManager;
    mTokenManager = tokenManager;
  }

  @NonNull
  public Observable<AdRequest> createAdRequest(@NonNull final String adUnitId,
                                               @NonNull final AdRequestParameters adRequestParameters) {
    return Observable.zip(
      mDeviceInfo.getAdvertisingInfo(),
      mTokenManager.getOrGenerateTokens(),
      mLocationManager.getLastLocationAndStartUpdates(),
      new Function3<AdvertisingIdClient.Info, Map<String, String>, Optional<Location>, AdRequest>() {
        @Override
        public AdRequest apply(
          AdvertisingIdClient.Info info, Map<String, String> tokens, Optional<Location> locationOptional) {
          final AdRequest.Builder builder = new AdRequest.Builder(
            adUnitId,
            Constants.API_VERSION,
            Constants.SDK_VERSION,
            mDeviceInfo.getAppVersion(),
            info.getId(),
            info.isLimitAdTrackingEnabled(),
            mDeviceInfo.getVendorId(info.getId()),
            mDeviceInfo.getTimeZoneShortDisplayName(),
            mDeviceInfo.getLocale().getLanguage(),
            mDeviceInfo.getOrientation().toString(),
            mDeviceInfo.getScreenWidthPx(),
            mDeviceInfo.getScreenHeightPx(),
            mDeviceInfo.getBrowserAgent(),
            mDeviceInfo.getModel(),
            mDeviceInfo.getConnectivity().toString(),
            mDeviceInfo.getCarrierName(),
            mDeviceInfo.getCarrierMCCMNC(),
            mSessionDepthManager.getSession(),
            tokens,
            mLocationManager.getLocationTrackingStatus().toString());

          if (locationOptional.isPresent()) {
            final Location location = locationOptional.get();
            builder.withLatitude(location.getLatitude());
            builder.withLongitude(location.getLongitude());

            final SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'", Locale.ENGLISH);
            simpleDateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
            final String locationTimestamp = simpleDateFormat.format(new Date(location.getTime()));

            builder.withLocationTimestamp(locationTimestamp);

            if (location.hasAccuracy()) {
              builder.withHorizontalAccuracy((double) location.getAccuracy());
            }
          }

          builder.withPriceFloorCents(adRequestParameters.getPriceFloorCents());

          return builder.build();
        }
      });
  }
}
