package io.maxads.ads.base;

import android.Manifest;
import android.annotation.SuppressLint;
import android.content.Context;
import android.content.pm.PackageManager;
import android.content.res.Configuration;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.os.Build;
import android.os.Looper;
import android.provider.Settings;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.v4.content.ContextCompat;
import android.telephony.TelephonyManager;
import android.text.TextUtils;
import android.webkit.WebSettings;
import android.webkit.WebView;

import com.google.android.gms.ads.identifier.AdvertisingIdClient;

import java.util.Locale;
import java.util.TimeZone;
import java.util.concurrent.Callable;

import io.maxads.ads.base.util.MaxAdsLog;
import io.reactivex.Observable;
import io.reactivex.ObservableSource;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.schedulers.Schedulers;
import okio.ByteString;

public class DeviceInfo {

  public enum Orientation {
    PORTRAIT("portrait"),
    LANDSCAPE("landscape"),
    NONE("none");

    @NonNull
    private final String mOrientation;

    Orientation(@NonNull String orientation) {
      mOrientation = orientation;
    }

    @Override
    public String toString() {
      return mOrientation;
    }
  }

  public enum Connectivity {
    ETHERNET("ethernet"),
    WIFI("wifi"),
    WWAN("wwan"),
    NONE("none");

    @NonNull
    private final String mConnectivity;

    Connectivity(@NonNull String connectivity) {
      mConnectivity = connectivity;
    }

    @Override
    public String toString() {
      return mConnectivity;
    }
  }

  @NonNull private static final String TAG = DeviceInfo.class.getSimpleName();
  @NonNull private static final String UNKNOWN_APP_VERSION_IDENTIFIER = "UNKNOWN";
  @NonNull private final Context mContext;
  @Nullable private String mUserAgent;
  @Nullable private final ConnectivityManager mConnectivityManager;
  @Nullable private final TelephonyManager mTelephonyManager;

  public DeviceInfo(@NonNull Context context) {
    mContext = context.getApplicationContext();
    mConnectivityManager = (ConnectivityManager) mContext.getSystemService(Context.CONNECTIVITY_SERVICE);
    mTelephonyManager = (TelephonyManager) mContext.getSystemService(Context.TELEPHONY_SERVICE);
  }

  /**
   * Attempt to use the play services advertising ID, but fall back on the old style Android ID.
   * https://developer.android.com/training/articles/user-data-ids.html
   * https://support.google.com/googleplay/android-developer/answer/6048248?hl=en
   * https://play.google.com/about/monetization-ads/ads/ad-id/
   */
  @SuppressLint("HardwareIds")
  @NonNull
  public Observable<AdvertisingIdClient.Info> getAdvertisingInfo() {
    return Observable.defer(new Callable<ObservableSource<AdvertisingIdClient.Info>>() {
      @Override
      public ObservableSource<AdvertisingIdClient.Info> call() {
        try {
          final AdvertisingIdClient.Info advertisingIdInfo = AdvertisingIdClient.getAdvertisingIdInfo(mContext);
          return Observable.just(advertisingIdInfo);
        } catch (Exception ignored) {
        }
        return Observable.just(new AdvertisingIdClient.Info(
          Settings.Secure.getString(mContext.getContentResolver(), Settings.Secure.ANDROID_ID), false));
      }
    }).subscribeOn(Schedulers.computation())
      .observeOn(AndroidSchedulers.mainThread());
  }

  @NonNull
  public String getAppVersion() {
    try {
      return mContext.getPackageManager().getPackageInfo(mContext.getPackageName(), 0).versionName;
    } catch (PackageManager.NameNotFoundException e) {
      MaxAdsLog.d(TAG, "Could not determine app version", e);
      return UNKNOWN_APP_VERSION_IDENTIFIER;
    }
  }

  @NonNull
  public String getVendorId(@Nullable String ifa) {
    if (ifa == null) {
      return "";
    }

    try {
      return ByteString.encodeUtf8(ifa).sha1().hex();
    } catch (Exception e) {
      MaxAdsLog.d(TAG, "Could not generate vendor id", e);
      return "";
    }
  }

  @NonNull
  public Locale getLocale() {
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N){
      return mContext.getResources().getConfiguration().getLocales().get(0);
    } else{
      // noinspection deprecation
      return mContext.getResources().getConfiguration().locale;
    }
  }

  @NonNull
  public String getTimeZoneShortDisplayName() {
    return TimeZone.getDefault().getDisplayName(false, TimeZone.SHORT);
  }

  @NonNull
  public Orientation getOrientation() {
    switch (mContext.getResources().getConfiguration().orientation) {
      case Configuration.ORIENTATION_PORTRAIT: {
        return Orientation.PORTRAIT;
      }
      case Configuration.ORIENTATION_LANDSCAPE: {
        return Orientation.LANDSCAPE;
      }
      default: {
        return Orientation.NONE;
      }
    }
  }

  public int getScreenWidthPx() {
    return mContext.getResources().getDisplayMetrics().widthPixels;
  }

  public int getScreenHeightPx() {
    return mContext.getResources().getDisplayMetrics().heightPixels;
  }

  /**
   * XXX: We initialize the user agent and catch exceptions in this method instead of in the constructor due to this bug
   * https://issuetracker.google.com/issues/37048374
   * https://stackoverflow.com/questions/29575313/namenotfoundexception-webview
   */
  @NonNull
  public String getBrowserAgent() {
    if (!TextUtils.isEmpty(mUserAgent)) {
      return mUserAgent;
    }

    try {
      if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
        mUserAgent = WebSettings.getDefaultUserAgent(mContext);
      } else if (Looper.myLooper() == Looper.getMainLooper()) {
        // Can only create WebViews on the main thread
        mUserAgent = new WebView(mContext).getSettings().getUserAgentString();
      }
    } catch (Exception ignore) {
      // May fail to get the user agent
    }

    if (TextUtils.isEmpty(mUserAgent)) {
      mUserAgent = System.getProperty("http.agent");
    }

    return mUserAgent;
  }

  @NonNull
  public String getModel() {
    return Build.MODEL;
  }

  @NonNull
  public Connectivity getConnectivity() {
    if (mConnectivityManager == null ||
      ContextCompat.checkSelfPermission(mContext, Manifest.permission.ACCESS_NETWORK_STATE)
        != PackageManager.PERMISSION_GRANTED) {
      return Connectivity.NONE;
    }

    final NetworkInfo activeNetworkInfo = mConnectivityManager.getActiveNetworkInfo();
    if (activeNetworkInfo == null) {
      return Connectivity.NONE;
    }

    switch (activeNetworkInfo.getType()) {
      case ConnectivityManager.TYPE_ETHERNET:
        return Connectivity.ETHERNET;
      case ConnectivityManager.TYPE_WIFI:
        return Connectivity.WIFI;
      case ConnectivityManager.TYPE_MOBILE:
      case ConnectivityManager.TYPE_MOBILE_DUN:
      case ConnectivityManager.TYPE_MOBILE_HIPRI:
      case ConnectivityManager.TYPE_MOBILE_MMS:
      case ConnectivityManager.TYPE_MOBILE_SUPL:
        return Connectivity.WWAN;
      default:
        return Connectivity.NONE;
    }
  }

  @NonNull
  public String getCarrierName() {
    if (mTelephonyManager == null) {
      return "";
    }

    try {
      return mTelephonyManager.getNetworkOperatorName();
    } catch (Exception ignored) {
    }

    return "";
  }

  @NonNull
  public String getCarrierMCCMNC() {
    if (mTelephonyManager == null) {
      return "";
    }

    String operator = "";
    try {
      // for getNetworkOperator, result may be unavailable on CDMA networks, hence we use getSimOperator in that case
      final String networkOperator = mTelephonyManager.getNetworkOperator();
      operator = TextUtils.isEmpty(networkOperator) ||
        (mTelephonyManager.getPhoneType() == TelephonyManager.PHONE_TYPE_CDMA &&
        mTelephonyManager.getSimState() == TelephonyManager.SIM_STATE_READY)
        ? mTelephonyManager.getSimOperator()
        : networkOperator;
    } catch (Exception ignored) {
    }

    String mccmnc = operator;
    // if operator string is mcc + mnc, then it will be a 6 digit code, otherwise don't format it
    if (operator.length() == 6) {
      mccmnc = operator.substring(0, 3) + "-" + operator.substring(3);
    }

    return mccmnc;
  }
}
