package io.maxads.ads.banner.controller;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;
import android.view.View;

import io.maxads.ads.banner.BannerDecorator;
import io.maxads.ads.banner.BannerDecoratorFactory;
import io.maxads.ads.banner.BannerRenderer;
import io.maxads.ads.banner.view.MaxBannerAdView;
import io.maxads.ads.base.MaxAds;
import io.maxads.ads.base.RefreshTimer;
import io.maxads.ads.base.SessionDepthManager;
import io.maxads.ads.base.api.AdRequestParameters;
import io.maxads.ads.base.api.MaxRequestManager;
import io.maxads.ads.base.cache.AdCache;
import io.maxads.ads.base.model.Ad;
import io.maxads.ads.base.util.Checks;
import io.maxads.ads.base.util.Helpers;
import io.maxads.ads.base.util.MaxAdsLog;
import io.maxads.ads.interstitial.MaxInterstitial;

/**
 * Manages the lifecycle of banners. This is the counterpart to {@link MaxInterstitial}
 */
public class BannerController implements MaxRequestManager.RequestListener, BannerDecorator.Listener {
  @NonNull private static final String TAG = BannerController.class.getSimpleName();

  @NonNull private final MaxBannerAdView mMaxBannerAdView;
  @NonNull private final BannerDecoratorFactory mBannerFactory;
  @NonNull private final MaxRequestManager mMaxRequestManager;
  @NonNull private final BannerRenderer mBannerRenderer;
  @NonNull private final SessionDepthManager mSessionDepthManager;
  @NonNull private final AdCache mAdCache;
  @NonNull private final Helpers.InitializationHelper mInitializationHelper;

  @Nullable private BannerDecorator mCurrentBannerDecorator;
  @Nullable private BannerDecorator mNextBannerDecorator;
  @Nullable private MaxBannerAdView.Listener mListener;
  private boolean mIsDestroyed;
  private boolean mDisableAutoRefresh;

  public BannerController(@NonNull Context context, @NonNull MaxBannerAdView maxBannerAdView) {
    this(maxBannerAdView, new BannerDecoratorFactory(context), new MaxRequestManager(), new BannerRenderer(),
      MaxAds.getSessionDepthManager(), MaxAds.getAdCache(), new Helpers.InitializationHelper());
  }

  @VisibleForTesting
  BannerController(@NonNull MaxBannerAdView maxBannerAdView,
                   @NonNull BannerDecoratorFactory bannerFactory,
                   @NonNull MaxRequestManager maxRequestManager,
                   @NonNull BannerRenderer bannerRenderer,
                   @NonNull SessionDepthManager sessionDepthManager,
                   @NonNull AdCache adCache,
                   @NonNull Helpers.InitializationHelper initializationHelper) {
    mMaxBannerAdView = maxBannerAdView;
    mBannerFactory = bannerFactory;
    mMaxRequestManager = maxRequestManager;
    mBannerRenderer = bannerRenderer;
    mMaxRequestManager.setRequestListener(this);
    mSessionDepthManager = sessionDepthManager;
    mAdCache = adCache;
    mInitializationHelper = initializationHelper;
  }

  public void setListener(@Nullable MaxBannerAdView.Listener listener) {
    mListener = listener;
  }

  public void setRequestListener(@Nullable MaxRequestManager.RequestListener listener) {
    mMaxRequestManager.setRequestListener(listener);
  }

  public void load(@NonNull String adUnitId) {
    load(adUnitId, AdRequestParameters.EMPTY_PARAMETERS);
  }

  public void load(@NonNull String adUnitId, @NonNull AdRequestParameters adRequestParameters) {
    if (!mInitializationHelper.isInitialized()) {
      return;
    }

    if (!Checks.NoThrow.checkNotNull(adUnitId, "adUnitId cannot be null")) {
      return;
    }

    if (!Checks.NoThrow.checkNotNull(adRequestParameters, "adRequestParameters cannot be null")) {
      return;
    }

    if (!Checks.NoThrow.checkArgument(!mIsDestroyed, TAG + " is destroyed")) {
      return;
    }

    mMaxRequestManager.requestAd(adUnitId, adRequestParameters);
    mMaxRequestManager.stopRefreshTimer();
  }

  @VisibleForTesting
  public void showAd(@NonNull Ad ad) {
    if (mIsDestroyed) {
      return;
    }

    // TODO (steffan): there is a low probability bug here if ads are requested rapidly that the mNextBanner
    // will continue to change before it can be loaded into the view. This means that there will be Banners
    // without a strong reference to them attempting to be loaded. It's possible for them to be garbage collected before
    // being displayed
    mNextBannerDecorator = mBannerFactory.createBanner(ad, this);
    if (mNextBannerDecorator == null) {
      startRefreshTimer(ad.getRefreshTimeSeconds());

      if (mListener != null) {
        mListener.onBannerError(mMaxBannerAdView);
      }
      return;
    }
    mNextBannerDecorator.load();
  }

  public void disableAutoRefresh(boolean disableAutoRefresh) {
    MaxAdsLog.d(TAG, "Auto refresh is disabled.");
    mDisableAutoRefresh = disableAutoRefresh;
  }

  public void startRefreshTimer(long delaySeconds) {
    if (mDisableAutoRefresh) {
      return;
    }

    mMaxRequestManager.startRefreshTimer(delaySeconds);
  }

  public void destroy() {
    mMaxRequestManager.destroy();
    destroyBannerDecorator(mCurrentBannerDecorator);
    mCurrentBannerDecorator = null;
    destroyBannerDecorator(mNextBannerDecorator);
    mNextBannerDecorator = null;
    mListener = null;
    mIsDestroyed = true;
  }

  private void destroyBannerDecorator(@Nullable BannerDecorator banner) {
    if (banner != null) {
      banner.destroy();
    }
  }

  // MaxRequestManager.RequestListener

  @Override
  public void onRequestSuccess(@NonNull Ad ad) {
    if (mIsDestroyed) {
      return;
    }

    // If we are controlling this ad through MAX then we don't need to keep it cached
    mAdCache.remove(ad.getAdUnitId());
    showAd(ad);
  }

  @Override
  public void onRequestFail(@NonNull Throwable throwable) {
    if (mIsDestroyed) {
      return;
    }

    startRefreshTimer(RefreshTimer.DEFAULT_REFRESH_TIME_SECONDS);

    if (mListener != null) {
      mListener.onBannerError(mMaxBannerAdView);
    }
  }

  // Banner.Listener

  @Override
  public void onBannerLoaded(@NonNull BannerDecorator bannerDecorator, @NonNull View view) {
    if (mIsDestroyed) {
      return;
    }

    mSessionDepthManager.incrementMaxSessionDepth(bannerDecorator.getAd().getAdUnitId());

    destroyBannerDecorator(mCurrentBannerDecorator);
    mCurrentBannerDecorator = mNextBannerDecorator;
    mNextBannerDecorator = null;

    startRefreshTimer(bannerDecorator.getAd().getRefreshTimeSeconds());
    mBannerRenderer.render(mMaxBannerAdView, view);

    if (mListener != null) {
      mListener.onBannerLoaded(mMaxBannerAdView);
    }
  }

  @Override
  public void onBannerClicked(@NonNull BannerDecorator bannerDecorator) {
    if (mIsDestroyed) {
      return;
    }

    if (mListener != null) {
      mListener.onBannerClicked(mMaxBannerAdView);
    }
  }

  @Override
  public void onBannerError(@NonNull BannerDecorator bannerDecorator) {
    if (mIsDestroyed) {
      return;
    }

    startRefreshTimer(bannerDecorator.getAd().getRefreshTimeSeconds());

    if (mListener != null) {
      mListener.onBannerError(mMaxBannerAdView);
    }
  }
}
