package io.maxads.ads.interstitial.vast3.presenter;

import android.content.Context;
import android.media.MediaPlayer;
import android.os.Bundle;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;
import android.text.TextUtils;

import io.maxads.ads.base.UrlHandlerDelegate;
import io.maxads.ads.base.util.MaxAdsLog;
import io.maxads.ads.interstitial.InterstitialBroadcastReceiver;
import io.maxads.ads.interstitial.InterstitialBroadcastSender;
import io.maxads.ads.interstitial.vast3.VASTApiClientDecorator;
import io.maxads.ads.interstitial.vast3.VASTError;
import io.maxads.ads.interstitial.vast3.VASTLastFramePresenter;
import io.maxads.ads.interstitial.vast3.VASTMacroDataImpl;
import io.maxads.ads.interstitial.vast3.VASTProgressTimerListener;
import io.maxads.ads.interstitial.vast3.VASTStartTimerListener;
import io.maxads.ads.interstitial.vast3.VASTTimer;
import io.maxads.ads.interstitial.vast3.VASTUITimerListener;
import io.maxads.ads.interstitial.vast3.activity.VASTActivity;
import io.maxads.ads.interstitial.vast3.model.VASTEventTracker;
import io.maxads.ads.interstitial.vast3.model.VASTVideoConfig;
import io.maxads.ads.interstitial.vast3.view.VASTViewModule;

public class VASTPresenter implements VASTViewModule.Listener, VASTTimer.Listener {
  @NonNull private static final String TAG = VASTPresenter.class.getSimpleName();

  public interface Listener {
    void onFinish();
  }

  @NonNull private final VASTViewModule mVASTViewModule;
  @NonNull private final VASTIconPresenter mVASTIconPresenter;
  @NonNull private final VASTCompanionAdPresenter mVASTCompanionAdPresenter;
  @NonNull private final VASTApiClientDecorator mVASTApiClientDecorator;
  @NonNull private final VASTVideoConfig mVASTVideoConfig;
  @NonNull private final InterstitialBroadcastSender mInterstitialBroadcastSender;
  @NonNull private final VASTTimer mVASTTimer;
  @NonNull private final VASTLastFramePresenter mVASTLastFramePresenter;
  @NonNull private final UrlHandlerDelegate mUrlHandlerDelegate;

  @Nullable private Listener mListener;

  public VASTPresenter(@NonNull Context context,
                       @NonNull VASTViewModule vastViewModule,
                       @NonNull VASTIconPresenter vastIconPresenter,
                       @NonNull VASTCompanionAdPresenter vastCompanionAdPresenter,
                       @NonNull VASTApiClientDecorator vastApiClientDecorator,
                       @NonNull VASTVideoConfig vastVideoConfig,
                       @NonNull InterstitialBroadcastSender interstitialBroadcastSender) {
    this(vastViewModule, vastIconPresenter, vastCompanionAdPresenter, vastApiClientDecorator,
      vastVideoConfig, interstitialBroadcastSender, new VASTTimer(vastViewModule, vastVideoConfig),
      new VASTLastFramePresenter(vastViewModule, vastVideoConfig), new UrlHandlerDelegate(context));
  }

  VASTPresenter(@NonNull VASTViewModule vastViewModule,
                @NonNull VASTIconPresenter vastIconPresenter,
                @NonNull VASTCompanionAdPresenter vastCompanionAdPresenter,
                @NonNull VASTApiClientDecorator vastApiClientDecorator,
                @NonNull VASTVideoConfig vastVideoConfig,
                @NonNull InterstitialBroadcastSender interstitialBroadcastSender,
                @NonNull VASTTimer vastTimer,
                @NonNull VASTLastFramePresenter vastLastFramePresenter,
                @NonNull UrlHandlerDelegate urlHandlerDelegate) {
    mVASTViewModule = vastViewModule;
    mVASTViewModule.setListener(this);
    mVASTIconPresenter = vastIconPresenter;
    mVASTIconPresenter.loadIcon();
    mVASTCompanionAdPresenter = vastCompanionAdPresenter;
    mVASTCompanionAdPresenter.loadCompanionAd();
    mVASTApiClientDecorator = vastApiClientDecorator;
    mVASTVideoConfig = vastVideoConfig;
    mInterstitialBroadcastSender = interstitialBroadcastSender;
    mVASTTimer = vastTimer;
    mVASTLastFramePresenter = vastLastFramePresenter;
    mUrlHandlerDelegate = urlHandlerDelegate;

    mVASTTimer.addListener(new VASTProgressTimerListener(mVASTApiClientDecorator,
      mVASTVideoConfig.getAbsoluteTrackers(), mVASTVideoConfig.getFractionalTrackers()));
    mVASTTimer.addListener(new VASTUITimerListener(mVASTViewModule, mVASTVideoConfig));
    mVASTTimer.addListener(new VASTStartTimerListener(mVASTApiClientDecorator, mVASTVideoConfig,
      mInterstitialBroadcastSender));
    mVASTTimer.addListener(this);
  }

  public void setListener(@Nullable Listener listener) {
    mListener = listener;
  }

  public void prepare() {
    if (mVASTVideoConfig.isVideoCompleted()) {
      mVASTLastFramePresenter.showLastFrame();
      allowVideoInteraction();
      mVASTCompanionAdPresenter.showCompanionAd();
      return;
    }

    final String url = mVASTVideoConfig.getMediaFileDiskUrl();
    if (mVASTVideoConfig.isVideoErrored() || TextUtils.isEmpty(url)) {
      finish();
      return;
    }

    mVASTViewModule.setVideoPath(url);
  }

  public void stop() {
    mVASTVideoConfig.setCurrentVideoPositionMs(mVASTViewModule.videoGetCurrentPositionMs());
    destroy();
  }

  public void onSaveInstanceState(@NonNull Bundle outState) {
    outState.putSerializable(VASTActivity.VAST_VIDEO_CONFIG_KEY, mVASTVideoConfig);
  }

  public void onBackPressed() {
    if (mVASTVideoConfig.isVideoCloseable()) {
      onCloseButtonClicked();
    }
  }

  public void destroy() {
    mVASTViewModule.stopVideo();
    mVASTTimer.stopTimer(false);
  }

  @VisibleForTesting
  void finish() {
    if (mListener != null) {
      mListener.onFinish();
    }

    mInterstitialBroadcastSender.sendBroadcast(InterstitialBroadcastReceiver.Action.DISMISS);
  }

  @VisibleForTesting
  void allowVideoInteraction() {
    mVASTVideoConfig.setVideoCloseable(true);
    mVASTViewModule.showCloseButton();
    mVASTViewModule.setVideoClickable(true);
  }

  // VASTViewModule.Listener

  @Override
  public void onMediaPlayerPrepared(@NonNull MediaPlayer mediaPlayer) {
    if (mVASTVideoConfig.isVideoCompletedOrErrored()) {
      return;
    }

    mVASTLastFramePresenter.prepareLastFrame(mVASTViewModule.videoGetDurationMs());

    // The video can only only be in this state in this code block if it was started and then paused previously
    if (mVASTVideoConfig.isVideoStarted()) {
      mVASTApiClientDecorator.fireVASTTrackers(mVASTVideoConfig.getEventTrackers(VASTEventTracker.Event.resume));
    }

    int currentVideoPositionMs = mVASTVideoConfig.getCurrentVideoPositionMs();
    if (currentVideoPositionMs > 0) {
      mVASTViewModule.videoSeekTo(currentVideoPositionMs);
    }

    mVASTViewModule.startVideo();

    mVASTTimer.startTimer();
    mVASTIconPresenter.showIcon();
  }

  @Override
  public void onMediaPlayerCompletion(@NonNull MediaPlayer mediaPlayer) {
    onVideoComplete(false);
  }

  @VisibleForTesting
  void onVideoComplete(boolean maxDurationExceeded) {
    mVASTTimer.stopTimer(true);
    mVASTLastFramePresenter.showLastFrame();

    allowVideoInteraction();

    if (!mVASTVideoConfig.isVideoCompletedOrErrored()) {
      mVASTVideoConfig.setVideoCompleted(true);
      if (!maxDurationExceeded) {
        mVASTApiClientDecorator.fireVASTTrackers(mVASTVideoConfig.getEventTrackers(VASTEventTracker.Event.complete));
      }

      mVASTIconPresenter.hideIcon();
      mVASTCompanionAdPresenter.showCompanionAd();
    }
  }

  @Override
  public void onMediaPlayerError(@NonNull MediaPlayer mediaPlayer, int what, int extra) {
    MaxAdsLog.w(TAG, "Media Player errors: WHAT:" + what + ": EXTRA:" + extra + ":");

    final VASTError vastError = VASTError.GENERAL_LINEAR_AD_ERROR;
    mVASTApiClientDecorator.fireVASTTrackers(mVASTVideoConfig.getErrorTrackers(), VASTMacroDataImpl.from(vastError));
    MaxAdsLog.w(TAG, vastError.getErrorCode());

    mVASTVideoConfig.setVideoErrored(true);
    finish();
  }

  @Override
  public void onVideoClicked() {
    final String clickThroughUrl = mVASTVideoConfig.getClickThroughUrl();

    // Before we send the app to the click through url, we will process ClickTracking URL's.
    mVASTApiClientDecorator.fireVASTTrackers(mVASTVideoConfig.getClickTrackers());

    // Navigate to the click through url
    if (!mUrlHandlerDelegate.handleUrl(clickThroughUrl)) {
      MaxAdsLog.w(TAG, "VAST click through URI unresolvable");
    }

    mInterstitialBroadcastSender.sendBroadcast(InterstitialBroadcastReceiver.Action.CLICK);
  }

  @Override
  public void onCloseButtonClicked() {
    destroy();

    if (!mVASTVideoConfig.isVideoCompleted()) {
      mVASTApiClientDecorator.fireVASTTrackers(mVASTVideoConfig.getEventTrackers(VASTEventTracker.Event.pause));
    }

    if (!mVASTVideoConfig.isVideoErrored()) {
      mVASTApiClientDecorator.fireVASTTrackers(mVASTVideoConfig.getEventTrackers(VASTEventTracker.Event.close));
      mVASTApiClientDecorator.fireVASTTrackers(mVASTVideoConfig.getEventTrackers(VASTEventTracker.Event.closeLinear));
    }

    finish();
  }

  // VASTTimer.Listener

  @Override
  public void onTick(int videoDurationMs, int currentPositionMs) {
  }

  @Override
  public void onStopped(int videoDurationMs, boolean videoCompleted, boolean maxDurationExceeded) {
  }

  @Override
  public void onMaxDurationExceeded() {
    mVASTViewModule.stopVideo();
    mVASTApiClientDecorator.fireVASTTrackers(mVASTVideoConfig.getEventTrackers(VASTEventTracker.Event.pause));
    onVideoComplete(true);
  }
}
