package io.maxads.ads.interstitial.vast3.model;

import android.content.Context;
import android.graphics.Point;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;
import android.util.DisplayMetrics;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import io.maxads.ads.base.util.MaxAdsLog;
import io.maxads.ads.base.util.Util;
import io.maxads.ads.interstitial.vast3.xml_model.ResourceXml;

public class VASTResource implements Serializable {
  private static final long serialVersionUID = 0L;
  @NonNull private static final String TAG = VASTResource.class.getSimpleName();

  @NonNull private static final List<String> VALID_CREATIVE_IMAGE_TYPES
          = Arrays.asList("image/jpeg", "image/png", "image/bmp", "image/gif");

  @NonNull private static final List<String> VALID_CREATIVE_APPLICATION_TYPES
          = Collections.singletonList("application/x-javascript");

  /**
   * Resource type ordered by priority
   */
  public enum Type {
    HTMLResource,
    StaticResource,
    IFrameResource,
  }

  /**
   * The type of the static resource. Only static resources only will have values other than NONE.
   */
  public enum CreativeType {
    NONE,
    IMAGE,
    JAVASCRIPT
  }

  @NonNull private final String mResource;
  @NonNull private final Type mType;
  @NonNull private final CreativeType mCreativeType;

  @Nullable
  public static VASTResource from(@Nullable ResourceXml resourceXml) {
    if (resourceXml == null) {
      return null;
    }

    final String resource;
    final VASTResource.Type type;
    final CreativeType creativeType;
    if (resourceXml.getHTMLResource() != null) {
      resource = resourceXml.getHTMLResource();
      type = VASTResource.Type.HTMLResource;
      creativeType = CreativeType.NONE;

    } else if (resourceXml.getStaticResource() != null) {
      resource = resourceXml.getStaticResource().value;
      type = VASTResource.Type.StaticResource;
      final String staticResourceCreativeType = resourceXml.getStaticResource().creativeType;
      creativeType = from(staticResourceCreativeType);
      if (creativeType == null) {
        MaxAdsLog.d(TAG, "Invalid static resource creativeType: " + staticResourceCreativeType);
        return null;
      }

    } else if (resourceXml.getIFrameResource() != null){
      resource = resourceXml.getIFrameResource();
      type = VASTResource.Type.IFrameResource;
      creativeType = CreativeType.NONE;

    } else {
      MaxAdsLog.d(TAG, "Resource does not have a valid type");
      return null;
    }

    return new VASTResource(resource, type, creativeType);
  }

  @Nullable
  private static CreativeType from(@Nullable String creativeType) {
    if (creativeType == null) {
      return null;
    }

    final String creativeTypeLower = creativeType.toLowerCase();
    if (VALID_CREATIVE_IMAGE_TYPES.contains(creativeTypeLower)) {
      return CreativeType.IMAGE;
    } else if (VALID_CREATIVE_APPLICATION_TYPES.contains(creativeTypeLower)) {
      return CreativeType.JAVASCRIPT;
    }

    return null;
  }

  @VisibleForTesting
  public VASTResource(@NonNull String resource, @NonNull Type type, @NonNull CreativeType creativeType) {
    mResource = resource.trim();
    mType = type;
    mCreativeType = creativeType;
  }

  @NonNull
  public String getResource() {
    return mResource;
  }

  @NonNull
  public Type getType() {
    return mType;
  }

  @NonNull
  public CreativeType getCreativeType() {
    return mCreativeType;
  }

  @Nullable
  public String getHtmlPayload(int widthDp, int heightDp) {
    switch (mType) {
      case HTMLResource:
        return mResource;
      case StaticResource:
        switch (mCreativeType) {
          case IMAGE:
            return "<html>" +
              "<head>" +
              "</head>" +
              // Set margin and padding to 0 in order to get rid of Android WebView default padding
              "<body style=\"margin:0; padding:0\">" +
                "<img src=\"" + mResource + "\" width=\"100%\" style=\"max-width:100%;max-height:100%;\" />" +
              "</body>" +
              "</html>";
          case JAVASCRIPT:
            return "<html>" +
              "<head>" +
              "</head>" +
              // Set margin and padding to 0 in order to get rid of Android WebView default padding
              "<body style=\"margin:0; padding:0\">" +
                "<script src=\"" + mResource + "\" />" +
              "</body>" +
              "</html>";
        }
        break;

      case IFrameResource:
        return "<html>" +
          "<head>" +
          "</head>" +
          // Set margin and padding to 0 in order to get rid of Android WebView default padding
          "<body style=\"margin:0; padding:0\">" +
            "<iframe frameborder=\"0\" scrolling=\"no\" marginwidth=\"0\" marginheight=\"0\"" +
              "style=\"border: 0px; margin: 0px; padding:0px\" width=\"" + widthDp + "\" height=\"" + heightDp + "\"" +
              "src=\"" + mResource + "\" />" +
          "</body>" +
          "</html>";
    }

    return null;
  }

  @NonNull
  public Point getScaledDimensionsPx(@NonNull Context context, int widthDp, int heightDp) {
    final DisplayMetrics metrics = context.getResources().getDisplayMetrics();
    final int deviceWidthPx = metrics.widthPixels;
    final int deviceHeightPx = metrics.heightPixels;
    final int landscapeWidthPx = Math.max(deviceWidthPx, deviceHeightPx);
    final int landscapeHeightPx = Math.min(deviceWidthPx, deviceHeightPx);

    final int widthPx = Util.dipsToIntPixels(widthDp, context);
    final int heightPx = Util.dipsToIntPixels(heightDp, context);
    final Point defaultPoint = new Point(widthPx, heightPx);

    // Return if the width and height already fit in the screen
    if (widthPx <= landscapeWidthPx && heightPx <= landscapeHeightPx) {
      return defaultPoint;
    }

    final Point point = new Point();
    if (VASTResource.Type.HTMLResource == mType) {
      point.x = Math.min(landscapeWidthPx, widthPx);
      point.y = Math.min(landscapeHeightPx, heightPx);
    } else {
      float widthRatio = (float) widthPx / landscapeWidthPx;
      float heightRatio = (float) heightPx / landscapeHeightPx;

      if (widthRatio >= heightRatio) {
        point.x = landscapeWidthPx;
        point.y = (int) (heightPx / widthRatio);
      } else {
        point.x = (int) (widthPx / heightRatio);
        point.y = landscapeHeightPx;
      }
    }

    return point;
  }
}
