package io.maxads.ads.interstitial.vast3;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;
import android.text.TextUtils;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

import io.maxads.ads.base.util.MaxAdsLog;

public class VASTMacroProcessor {
  @NonNull private static final String TAG = VASTMacroProcessor.class.getSimpleName();

  /**
   * Supported VAST URL macros. MAX will find and replace macros that follow the format of [name].
   * Example: "https://www.website.com/path?error=[ERRORCODE]" becomes "https://www.website.com/path?error=100"
   */
  public enum VASTMacro {
    /**
     * VAST error number. See {@link VASTError}
     */
    ERRORCODE,

    /**
     * Current progress of the video in the format HH:MM:SS.mmm
     */
    CONTENTPLAYHEAD,

    /**
     * Random 8-digit number for cache busting
     */
    CACHEBUSTING,

    /**
     * Video URL
     */
    ASSETURI
  }

  @NonNull
  private static String getCachebustingString() {
    return String.format(Locale.US, "%08d", Math.round(Math.random() * 100000000));
  }

  @NonNull private final Map<VASTMacro, String> mMacroMap;

  public VASTMacroProcessor() {
    this(getCachebustingString());
  }

  VASTMacroProcessor(@NonNull String cacheBustingString) {
    mMacroMap = new HashMap<>();
    mMacroMap.put(VASTMacro.CACHEBUSTING, cacheBustingString);
  }

  @NonNull
  public String processUrl(@NonNull String url) {
    if (TextUtils.isEmpty(url)) {
      return url;
    }

    String macroUrl = url;

    for (final VASTMacro vastMacro : VASTMacro.values()) {
      String value = mMacroMap.get(vastMacro);
      if (value == null) {
        value = "";
      }

      macroUrl = macroUrl.replaceAll("\\[" + vastMacro.name() + "]", value);
    }

    return macroUrl;
  }

  @NonNull
  public VASTMacroProcessor withVASTError(@Nullable VASTError vastError) {
    if (vastError == null) {
      return this;
    }

    mMacroMap.put(VASTMacro.ERRORCODE, vastError.getErrorCode());
    return this;
  }

  @NonNull
  public VASTMacroProcessor withContentPlayHeadMs(@Nullable Integer contentPlayHeadMs) {
    if (contentPlayHeadMs == null) {
      return this;
    }

    final String formattedContentPlayHead = VASTUtil.formatTimeMs(contentPlayHeadMs);
    if (!TextUtils.isEmpty(formattedContentPlayHead)) {
      mMacroMap.put(VASTMacro.CONTENTPLAYHEAD, formattedContentPlayHead);
    }
    return this;
  }

  @NonNull
  public VASTMacroProcessor withAssetUri(@Nullable String assetUri) {
    if (TextUtils.isEmpty(assetUri)) {
      return this;
    }

    try {
      // URL-encode
      assetUri = URLEncoder.encode(assetUri, "UTF-8");
    } catch (UnsupportedEncodingException e) {
      MaxAdsLog.d(TAG, "Failed to encode url", e);
    }

    mMacroMap.put(VASTMacro.ASSETURI, assetUri);
    return this;
  }
}
