package io.maxads.ads.interstitial.vast3;

import android.annotation.SuppressLint;
import android.graphics.Bitmap;
import android.media.MediaMetadataRetriever;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;

import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;

import io.maxads.ads.base.util.MaxAdsLog;
import io.maxads.ads.base.util.Optional;
import io.maxads.ads.interstitial.vast3.model.VASTVideoConfig;
import io.maxads.ads.interstitial.vast3.view.VASTViewModule;
import io.reactivex.Observable;
import io.reactivex.ObservableSource;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.functions.Consumer;
import io.reactivex.schedulers.Schedulers;

/**
 * Fetch the last frame of the video to display when the video ends. This will be displayed when the video
 * naturally ends or when it exceeds its max duration.
 */
// TODO test
public class VASTLastFramePresenter {
  @NonNull private static final String TAG = VASTLastFramePresenter.class.getSimpleName();
  private static final int OFFSET_US = 200000;

  @NonNull private final VASTViewModule mVASTViewModule;
  @NonNull private final VASTVideoConfig mVASTVideoConfig;
  @NonNull private final MediaMetadataRetriever mMediaMetadataRetriever;
  @Nullable private Bitmap mLastFrame;
  private boolean mShowLastFrameCalled;

  public VASTLastFramePresenter(@NonNull VASTViewModule vastViewModule, @NonNull VASTVideoConfig vastVideoConfig) {
    mVASTViewModule = vastViewModule;
    mVASTVideoConfig = vastVideoConfig;
    mMediaMetadataRetriever = new MediaMetadataRetriever();
  }

  @SuppressLint("CheckResult")
  public void prepareLastFrame(int videoDurationMs) {
    if (mLastFrame != null || videoDurationMs <= 0) {
      return;
    }

    fetchLastFrame(videoDurationMs, mVASTVideoConfig.getMediaFileDiskUrl())
      .observeOn(AndroidSchedulers.mainThread())
      .subscribe(
        new Consumer<Optional<Bitmap>>() {
          @Override
          public void accept(Optional<Bitmap> bitmapOptional) {
            if (!bitmapOptional.isPresent()) {
              return;
            }

            mLastFrame = bitmapOptional.get();
            if (mShowLastFrameCalled) {
              showLastFrame();
            }
          }
        },
        new Consumer<Throwable>() {
          @Override
          public void accept(Throwable throwable) {
            MaxAdsLog.d(TAG, "Unable to fetch last frame of video", throwable);
          }
        });
  }

  @VisibleForTesting
  @NonNull
  Observable<Optional<Bitmap>> fetchLastFrame(final int videoDurationMs, final String mediaFileDiskUrl) {
    return Observable.defer(new Callable<ObservableSource<Optional<Bitmap>>>() {
      @Override
      public ObservableSource<Optional<Bitmap>> call() {
        mMediaMetadataRetriever.setDataSource(mediaFileDiskUrl);
        return Observable.just(
          Optional.of(
            // Get a frame just before the video ends. If we try to get a frame that's actually past the end of the
            // video or before 0, this will return some arbitrary frame.
            mMediaMetadataRetriever.getFrameAtTime(TimeUnit.MILLISECONDS.toMicros(videoDurationMs) - OFFSET_US,
              MediaMetadataRetriever.OPTION_CLOSEST)));
      }
    }).subscribeOn(Schedulers.computation());
  }

  public void showLastFrame() {
    if (mLastFrame != null) {
      mVASTViewModule.showLastVideoFrame(mLastFrame);
    }
    mShowLastFrameCalled = true;
  }
}
