package io.maxads.ads.interstitial.vast3;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.DisplayMetrics;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import io.maxads.ads.interstitial.vast3.model.VASTCompanionAdConfig;
import io.maxads.ads.interstitial.vast3.xml_model.CompanionAdXml;

public class CompanionAdPicker {
  @NonNull private final Context mContext;
  @NonNull private final AreaComparator mAreaComparator;

  public CompanionAdPicker(@NonNull Context context) {
    mContext = context;

    final DisplayMetrics metrics = mContext.getResources().getDisplayMetrics();
    final int deviceWidth = metrics.widthPixels;
    final int deviceHeight = metrics.heightPixels;
    final int deviceArea = deviceWidth * deviceHeight;

    mAreaComparator = new AreaComparator(deviceArea);
  }

  @Nullable
  public VASTCompanionAdConfig pickCompanionAd(@Nullable List<CompanionAdXml> companionAdXmls) {
    if (companionAdXmls == null) {
      return null;
    }

    final List<VASTCompanionAdConfig> validCompanionAds = prefilterCompanionAds(companionAdXmls);
    if (validCompanionAds.isEmpty()) {
      return null;
    }

    final List<VASTCompanionAdConfig> htmlCompanionAds = new ArrayList<>(validCompanionAds.size());
    final List<VASTCompanionAdConfig> staticCompanionAds = new ArrayList<>(validCompanionAds.size());
    final List<VASTCompanionAdConfig> iFrameCompanionAds = new ArrayList<>(validCompanionAds.size());

    for (VASTCompanionAdConfig vastCompanionAdConfig : validCompanionAds) {
      switch (vastCompanionAdConfig.getVASTResource().getType()) {
        case HTMLResource:
          htmlCompanionAds.add(vastCompanionAdConfig);
          break;
        case StaticResource:
          staticCompanionAds.add(vastCompanionAdConfig);
          break;
        case IFrameResource:
          iFrameCompanionAds.add(vastCompanionAdConfig);
          break;
      }
    }

    if (!htmlCompanionAds.isEmpty()) {
      Collections.sort(htmlCompanionAds, mAreaComparator);
      return htmlCompanionAds.get(0);
    } else if (!staticCompanionAds.isEmpty()) {
      Collections.sort(staticCompanionAds, mAreaComparator);
      return staticCompanionAds.get(0);
    } else {
      Collections.sort(iFrameCompanionAds, mAreaComparator);
      return iFrameCompanionAds.get(0);
    }
  }

  private List<VASTCompanionAdConfig> prefilterCompanionAds(@NonNull List<CompanionAdXml> companionAdXmls) {
    final List<VASTCompanionAdConfig> vastCompanionAdConfigs = new ArrayList<>(companionAdXmls.size());
    for (CompanionAdXml companionAdXml : companionAdXmls) {
      final VASTCompanionAdConfig vastCompanionAdConfig = VASTCompanionAdConfig.from(mContext, companionAdXml);
      if (vastCompanionAdConfig != null) {
        vastCompanionAdConfigs.add(vastCompanionAdConfig);
      }
    }

    return vastCompanionAdConfigs;
  }

  private static class AreaComparator implements Comparator<VASTCompanionAdConfig> {
    private final int mDeviceArea;

    private AreaComparator(int deviceArea) {
      mDeviceArea = deviceArea;
    }

    @Override
    public int compare(@NonNull VASTCompanionAdConfig obj1, @NonNull VASTCompanionAdConfig obj2) {
      // get area of the video of the two Companion Ads
      int obj1Area = obj1.getScaledWidthPx() * obj1.getScaledHeightPx();
      int obj2Area = obj2.getScaledWidthPx() * obj2.getScaledHeightPx();

      // get the difference between the area of the MediaFile and the area of the screen
      int obj1Diff = Math.abs(mDeviceArea - obj1Area);
      int obj2Diff = Math.abs(mDeviceArea - obj2Area);

      // choose the Companion Ad which has the lower difference in area
      if (obj1Diff < obj2Diff) {
        return -1;
      } else if (obj1Diff > obj2Diff) {
        return 1;
      } else {
        return 0;
      }
    }
  }
}
