package io.maxads.ads.interstitial;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;
import android.text.TextUtils;
import android.webkit.WebView;

import io.maxads.ads.base.Constants;
import io.maxads.ads.base.MaxAds;
import io.maxads.ads.base.cache.WebViewCache;
import io.maxads.ads.base.model.Ad;
import io.maxads.ads.base.mraid.internal.MRAIDHtmlProcessor;
import io.maxads.ads.base.mraid.view.MraidCacheWebViewClient;
import io.maxads.ads.base.mraid.view.MraidWebView;
import io.maxads.ads.base.util.MaxAdsLog;
import io.maxads.ads.interstitial.activity.MraidInterstitialActivity;

public class MraidInterstitial implements Interstitial, InterstitialBroadcastReceiver.Listener {
  @NonNull private static final String TAG = MraidInterstitial.class.getSimpleName();

  public static class MraidInterstitialFactory implements InterstitialFactory {
    @NonNull private final Activity mActivity;

    public MraidInterstitialFactory(@NonNull Activity activity) {
      mActivity = activity;
    }

    @Nullable
    @Override
    public Interstitial createInterstitial(@NonNull Ad ad) {
      return new MraidInterstitial(mActivity, ad);
    }
  }

  @NonNull private final Context mApplicationContext;
  @NonNull private final Ad mAd;
  @NonNull private final WebViewCache mWebViewCache;
  @NonNull private final InterstitialBroadcastReceiver mInterstitialBroadcastReceiver;

  @Nullable private Interstitial.Listener mListener;

  public MraidInterstitial(@NonNull Activity activity, @NonNull Ad ad) {
    this(activity, ad, MaxAds.getWebViewCache(), new InterstitialBroadcastReceiver(activity));
  }

  @VisibleForTesting
  MraidInterstitial(@NonNull Activity activity,
                    @NonNull Ad ad,
                    @NonNull WebViewCache webViewCache,
                    @NonNull InterstitialBroadcastReceiver interstitialBroadcastReceiver) {
    mApplicationContext = activity.getApplicationContext();
    mAd = ad;
    mWebViewCache = webViewCache;
    mInterstitialBroadcastReceiver = interstitialBroadcastReceiver;
    mInterstitialBroadcastReceiver.setListener(this);
  }

  @Override
  public void setListener(@Nullable Listener listener) {
    mListener = listener;
  }

  @Override
  public void load() {
    final String creative = mAd.getCreative();
    if (TextUtils.isEmpty(creative)) {
      MaxAdsLog.w(TAG, "Unable to load MRAID interstitial, creative was empty");
      notifyLoadFailed();
      return;
    }

    precacheWebView(new MraidWebView(mApplicationContext), creative);
  }

  @VisibleForTesting
  void precacheWebView(@NonNull MraidWebView mraidWebView, @NonNull String html) {
    mraidWebView.setWebViewClient(new MraidCacheWebViewClient() {
      @Override
      public void onPageFinished(WebView view, String url) {
        super.onPageFinished(view, url);
        webViewCacheSuccess();
      }

      @Override
      public void onReceivedError(final WebView view, final int errorCode, final String description,
                                  final String failingUrl) {
        super.onReceivedError(view, errorCode, description, failingUrl);
        webViewCacheFail();
      }
    });

    mWebViewCache.put(mInterstitialBroadcastReceiver.getBroadcastId(), new WebViewCache.Data(mraidWebView, this));
    mraidWebView.loadDataWithBaseURL(Constants.MAXADS_URL, MRAIDHtmlProcessor.processRawHtml(html),
      "text/html", "UTF-8", null);
  }

  @VisibleForTesting
  void webViewCacheSuccess() {
    if (mListener != null) {
      mListener.onInterstitialLoaded(MraidInterstitial.this);
    }
  }

  @VisibleForTesting
  void webViewCacheFail() {
    mWebViewCache.remove(mInterstitialBroadcastReceiver.getBroadcastId());
    notifyLoadFailed();
  }

  @Override
  public void show() {
    mInterstitialBroadcastReceiver.register();

    final Intent intent = new Intent(mApplicationContext, MraidInterstitialActivity.class);
    intent.putExtra(MraidInterstitialActivity.HTML_KEY, mAd.getCreative());
    intent.putExtra(MraidInterstitialActivity.BROADCAST_ID_KEY, mInterstitialBroadcastReceiver.getBroadcastId());
    intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
    mApplicationContext.startActivity(intent);
  }

  @Override
  public void destroy() {
    mWebViewCache.remove(mInterstitialBroadcastReceiver.getBroadcastId());
    mInterstitialBroadcastReceiver.destroy();
    mListener = null;
  }

  private void notifyLoadFailed() {
    if (mListener != null) {
      mListener.onInterstitialFailedToLoad(this);
    }
  }

  @Override
  public void onReceivedAction(@NonNull InterstitialBroadcastReceiver.Action action) {
    mInterstitialBroadcastReceiver.handleAction(action, this, mListener);
  }
}
