package io.maxads.ads.interstitial;

import android.app.Activity;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;

import io.maxads.ads.banner.controller.BannerController;
import io.maxads.ads.base.MaxAds;
import io.maxads.ads.base.SessionDepthManager;
import io.maxads.ads.base.api.AdRequestParameters;
import io.maxads.ads.base.api.MaxRequestManager;
import io.maxads.ads.base.cache.AdCache;
import io.maxads.ads.base.model.Ad;
import io.maxads.ads.base.util.Checks;
import io.maxads.ads.base.util.Helpers;
import io.maxads.ads.base.util.MaxAdsLog;

/**
 * Manages the lifecycle of interstitials. This is the counterpart to {@link BannerController}
 */
public class MaxInterstitial implements MaxRequestManager.RequestListener, InterstitialDecorator.Listener {
  @NonNull private static final String TAG = MaxInterstitial.class.getSimpleName();

  public interface Listener {
    /**
     * Called when an interstitial ad response is fully fetched including all the assets associated with the DSP
     * ad markup. For VAST this means the full video is downloaded and cached.
     */
    void onInterstitialLoaded(@NonNull MaxInterstitial maxInterstitial);

    /**
     * Called when an interstitial ad response or any of its assets fails to load.
     */
    void onInterstitialFailedToLoad(@NonNull MaxInterstitial maxInterstitial);

    /**
     * Called when the ad container is shown on screen to the user. For certain ad formats such as VAST,
     * the actual ad content may be displayed shortly after this call when it becomes ready to fill the container.
     */
    void onInterstitialShown(@NonNull MaxInterstitial maxInterstitial);

    /**
     * Called when the user taps an interstitial and the ad is about to perform its resulting action.
     */
    void onInterstitialClicked(@NonNull MaxInterstitial maxInterstitial);

    /**
     * Called immediately before an interstitial ad will be dismissed from the screen.
     */
    void onInterstitialDismissed(@NonNull MaxInterstitial maxInterstitial);

    /**
     * Called when a loaded interstitial ad is no longer eligible to be displayed.
     */
    void onInterstitialDidExpire(@NonNull MaxInterstitial maxInterstitial);

    /**
     * Called when the interstitial encounters issues other than loading errors.
     */
    void onInterstitialError(@NonNull MaxInterstitial maxInterstitial);
  }

  @NonNull private final InterstitialDecoratorFactory mInterstitialDecoratorFactory;
  @NonNull private final MaxRequestManager mMaxRequestManager;
  @NonNull private final SessionDepthManager mSessionDepthManager;
  @NonNull private final AdCache mAdCache;
  @NonNull private final Helpers.InitializationHelper mInitializationHelper;

  @Nullable private InterstitialDecorator mInterstitialDecorator;
  @Nullable private Listener mListener;
  private boolean mIsDestroyed;

  public MaxInterstitial(@NonNull Activity activity) {
    this(new InterstitialDecoratorFactory(activity), new MaxRequestManager(), MaxAds.getSessionDepthManager(),
      MaxAds.getAdCache(), new Helpers.InitializationHelper());
  }

  @VisibleForTesting
  MaxInterstitial(@NonNull InterstitialDecoratorFactory interstitialDecoratorFactory,
                  @NonNull MaxRequestManager maxRequestManager,
                  @NonNull SessionDepthManager sessionDepthManager,
                  @NonNull AdCache adCache,
                  @NonNull Helpers.InitializationHelper initializationHelper) {
    mInterstitialDecoratorFactory = interstitialDecoratorFactory;
    mMaxRequestManager = maxRequestManager;
    mMaxRequestManager.setRequestListener(this);
    mSessionDepthManager = sessionDepthManager;
    mAdCache = adCache;
    mInitializationHelper = initializationHelper;
  }

  public void setListener(@Nullable Listener listener) {
    mListener = listener;
  }

  public void setRequestListener(@Nullable MaxRequestManager.RequestListener listener) {
    mMaxRequestManager.setRequestListener(listener);
  }

  public void load(@NonNull String adUnitId) {
    load(adUnitId, AdRequestParameters.EMPTY_PARAMETERS);
  }

  public void load(@NonNull String adUnitId, @NonNull AdRequestParameters adRequestParameters) {
    if (!mInitializationHelper.isInitialized()) {
      return;
    }

    if (!Checks.NoThrow.checkNotNull(adUnitId, "adUnitId cannot be null")) {
      return;
    }

    if (!Checks.NoThrow.checkNotNull(adRequestParameters, "adRequestParameters cannot be null")) {
      return;
    }

    if (!Checks.NoThrow.checkArgument(!mIsDestroyed, TAG + " has been destroyed")) {
      return;
    }

    if (mMaxRequestManager.isLoading()) {
      MaxAdsLog.d(TAG, "Interstitial is loading");
      return;
    }

    // If this is non null it means we are already working with an interstitial
    if (mInterstitialDecorator != null) {
      mInterstitialDecorator.logState();
      return;
    }

    mMaxRequestManager.requestAd(adUnitId, adRequestParameters);
  }

  @VisibleForTesting
  void loadInterstitial(@NonNull Ad ad) {
    if (isDestroyed()) {
      return;
    }

    mInterstitialDecorator = mInterstitialDecoratorFactory.createInterstitial(ad, this);
    if (mInterstitialDecorator == null) {
      if (mListener != null) {
        mListener.onInterstitialError(this);
      }
      return;
    }

    mInterstitialDecorator.load();
  }

  public void show() {
    if (mInterstitialDecorator == null) {
      MaxAdsLog.d(TAG, "Interstitial is not loaded");
      return;
    }

    mInterstitialDecorator.show();
  }

  public void destroy() {
    mMaxRequestManager.destroy();
    destroyInterstitialDecorator();
    mListener = null;
    mIsDestroyed = true;
  }

  private void destroyInterstitialDecorator() {
    if (mInterstitialDecorator != null) {
      mInterstitialDecorator.destroy();
      mInterstitialDecorator = null;
    }
  }

  public boolean isReady() {
    return mInterstitialDecorator != null && mInterstitialDecorator.isReady();
  }

  public boolean isDestroyed() {
    return mIsDestroyed;
  }

  // MaxRequestManager.RequestListener

  @Override
  public void onRequestSuccess(@NonNull Ad ad) {
    if (isDestroyed()) {
      return;
    }

    // If we are controlling this ad through MAX then we don't need to keep it cached
    mAdCache.remove(ad.getAdUnitId());
    loadInterstitial(ad);
  }

  @Override
  public void onRequestFail(@NonNull Throwable throwable) {
    if (isDestroyed()) {
      return;
    }

    if (mListener != null) {
      mListener.onInterstitialFailedToLoad(this);
    }
  }

  // MaxInterstitial.Listener

  @Override
  public void onInterstitialLoaded(@NonNull InterstitialDecorator interstitialDecorator) {
    if (isDestroyed()) {
      return;
    }

    if (mListener != null) {
      mListener.onInterstitialLoaded(this);
    }
  }

  @Override
  public void onInterstitialFailedToLoad(@NonNull InterstitialDecorator interstitialDecorator) {
    if (isDestroyed()) {
      return;
    }

    destroyInterstitialDecorator();

    if (mListener != null) {
      mListener.onInterstitialFailedToLoad(this);
    }
  }

  @Override
  public void onInterstitialShown(@NonNull InterstitialDecorator interstitialDecorator) {
    if (isDestroyed()) {
      return;
    }

    if (mListener != null) {
      mListener.onInterstitialShown(this);
    }
  }

  @Override
  public void onInterstitialImpressed(@NonNull InterstitialDecorator interstitialDecorator) {
    if (isDestroyed()) {
      return;
    }

    mSessionDepthManager.incrementMaxSessionDepth(interstitialDecorator.getAd().getAdUnitId());
  }

  @Override
  public void onInterstitialClicked(@NonNull InterstitialDecorator interstitialDecorator) {
    if (isDestroyed()) {
      return;
    }

    if (mListener != null) {
      mListener.onInterstitialClicked(this);
    }
  }

  @Override
  public void onInterstitialDismissed(@NonNull InterstitialDecorator interstitialDecorator) {
    if (isDestroyed()) {
      return;
    }

    destroyInterstitialDecorator();

    if (mListener != null) {
      mListener.onInterstitialDismissed(this);
    }
  }

  @Override
  public void onInterstitialDidExpire(@NonNull InterstitialDecorator interstitialDecorator) {
    if (isDestroyed()) {
      return;
    }

    destroyInterstitialDecorator();

    if (mListener != null) {
      mListener.onInterstitialDidExpire(this);
    }
  }

  @Override
  public void onInterstitialError(@NonNull InterstitialDecorator interstitialDecorator) {
    if (isDestroyed()) {
      return;
    }

    destroyInterstitialDecorator();

    if (mListener != null) {
      mListener.onInterstitialError(this);
    }
  }

  @Deprecated
  @VisibleForTesting
  void setInterstitialDecorator(@NonNull InterstitialDecorator interstitialDecorator) {
    mInterstitialDecorator = interstitialDecorator;
  }
}
