package io.maxads.ads.interstitial;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;

import io.maxads.ads.base.util.Helpers;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.disposables.Disposable;
import io.reactivex.functions.Consumer;

/**
 * Timer for interstitials to complete their loading phase. If an interstitial exceeds this limit,
 * the {@link io.maxads.ads.interstitial.MaxInterstitial.Listener#onInterstitialFailedToLoad(MaxInterstitial)}
 * callback will be invoked and the interstitial destroyed.
 */
public class InterstitialLoadTimer {
  public interface Listener {
    void onTimerComplete();
  }

  private final long mLoadTimeoutMs;
  @NonNull private final Helpers.TimerFactory mTimerFactory;
  @Nullable private Disposable mDisposable;
  private boolean mDestroyed;

  @Nullable private Listener mListener;

  public InterstitialLoadTimer(long loadTimeoutMs) {
    this(loadTimeoutMs, new Helpers.TimerFactory());
  }

  @VisibleForTesting
  InterstitialLoadTimer(long loadTimeoutMs, @NonNull Helpers.TimerFactory timerFactory) {
    mLoadTimeoutMs = loadTimeoutMs;
    mTimerFactory = timerFactory;
  }

  public void setListener(@Nullable Listener listener) {
    mListener = listener;
  }

  public void start() {
    if (mDestroyed) {
      return;
    }

    // Subscribes on Schedulers.computation() by default
    mDisposable = mTimerFactory.createTimerMs(mLoadTimeoutMs)
      .observeOn(AndroidSchedulers.mainThread())
      .subscribe(new Consumer<Long>() {
        @Override
        public void accept(Long aLong) {
          if (mListener != null) {
            mListener.onTimerComplete();
          }
        }
      });
  }

  public void destroy() {
    doDestroy(mDisposable);
  }

  void doDestroy(@Nullable Disposable disposable) {
    if (disposable != null) {
      disposable.dispose();
    }
    mDestroyed = true;
  }
}
