package io.maxads.ads.interstitial;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;

import io.maxads.ads.base.model.Ad;
import io.maxads.ads.base.util.MaxAdsLog;

/**
 * Wraps implementations of interstitials and provides shared functionality.
 */
public class InterstitialDecorator implements Interstitial.Listener, InterstitialLoadTimer.Listener {
  @NonNull private static final String TAG = InterstitialDecorator.class.getSimpleName();

  public interface Listener {
    void onInterstitialLoaded(@NonNull InterstitialDecorator interstitialDecorator);
    void onInterstitialFailedToLoad(@NonNull InterstitialDecorator interstitialDecorator);
    void onInterstitialShown(@NonNull InterstitialDecorator interstitialDecorator);
    void onInterstitialImpressed(@NonNull InterstitialDecorator interstitialDecorator);
    void onInterstitialClicked(@NonNull InterstitialDecorator interstitialDecorator);
    void onInterstitialDismissed(@NonNull InterstitialDecorator interstitialDecorator);
    void onInterstitialDidExpire(@NonNull InterstitialDecorator interstitialDecorator);
    void onInterstitialError(@NonNull InterstitialDecorator interstitialDecorator);
  }

  @NonNull private final Interstitial mInterstitial;
  @NonNull private final Ad mAd;
  @NonNull private final InterstitialLoadTimer mInterstitialLoadTimer;
  @NonNull private final InterstitialState mInterstitialState;

  @Nullable private Listener mListener;

  public InterstitialDecorator(@NonNull Interstitial interstitial, @NonNull Ad ad) {
    this(interstitial, ad, new InterstitialLoadTimer(ad.getAdUnit().getInterstitialLoadTimeoutMs()),
      new InterstitialState(TAG));
  }

  @VisibleForTesting
  InterstitialDecorator(@NonNull Interstitial interstitial,
                        @NonNull Ad ad,
                        @NonNull InterstitialLoadTimer interstitialLoadTimer,
                        @NonNull InterstitialState interstitialState) {
    mInterstitial = interstitial;
    mInterstitial.setListener(this);
    mAd = ad;
    mInterstitialLoadTimer = interstitialLoadTimer;
    mInterstitialLoadTimer.setListener(this);
    mInterstitialState = interstitialState;
  }

  public void setListener(@Nullable Listener listener) {
    mListener = listener;
  }

  @NonNull
  public Ad getAd() {
    return mAd;
  }

  public void load() {
    if (!mInterstitialState.transitionState(InterstitialState.State.LOADING)) {
      return;
    }

    MaxAdsLog.d(TAG, "Loading interstitial for ad unit id: " + mAd.getAdUnitId());
    mAd.trackSelectedUrls();
    mInterstitial.load();
    mInterstitialLoadTimer.start();
  }

  public void show() {
    if (!mInterstitialState.transitionState(InterstitialState.State.SHOWING)) {
      return;
    }

    MaxAdsLog.d(TAG, "Showing interstitial for ad unit id: " + mAd.getAdUnitId());
    mInterstitial.show();
  }

  public void destroy() {
    MaxAdsLog.d(TAG, "Destroying interstitial for ad unit id: " + mAd.getAdUnitId());
    mInterstitial.destroy();
    mInterstitialLoadTimer.destroy();
    mListener = null;
    mInterstitialState.transitionState(InterstitialState.State.DESTROYED);
  }

  public boolean isReady() {
    return mInterstitialState.getState() == InterstitialState.State.READY;
  }

  public boolean isDestroyed() {
    return mInterstitialState.getState() == InterstitialState.State.DESTROYED;
  }

  public void logState() {
    mInterstitialState.logState();
  }

  // Interstitial.Listener

  public void onInterstitialLoaded(@NonNull Interstitial interstitial) {
    if (!mInterstitialState.transitionState(InterstitialState.State.READY)) {
      return;
    }

    mInterstitialLoadTimer.destroy();

    MaxAdsLog.d(TAG, "Interstitial loaded for ad unit id: " + mAd.getAdUnitId());
    if (mListener != null) {
      mListener.onInterstitialLoaded(this);
    }
  }

  @Override
  public void onInterstitialFailedToLoad(@NonNull Interstitial interstitial) {
    if (!mInterstitialState.transitionState(InterstitialState.State.IDLE)) {
      return;
    }

    mInterstitialLoadTimer.destroy();

    MaxAdsLog.d(TAG, "Interstitial failed to load for ad unit id: " + mAd.getAdUnitId());
    if (mListener != null) {
      mListener.onInterstitialFailedToLoad(this);
    }
  }

  @Override
  public void onInterstitialShown(@NonNull Interstitial interstitial) {
    if (isDestroyed()) {
      return;
    }

    MaxAdsLog.d(TAG, "Interstitial shown for ad unit id: " + mAd.getAdUnitId());
    if (mListener != null) {
      mListener.onInterstitialShown(this);
    }
  }

  @Override
  public void onInterstitialImpressed(@NonNull Interstitial interstitial) {
    if (isDestroyed()) {
      return;
    }

    MaxAdsLog.d(TAG, "Interstitial impressed for ad unit id: " + mAd.getAdUnitId());
    mAd.trackImpressionUrls();
    if (mListener != null) {
      mListener.onInterstitialImpressed(this);
    }
  }

  @Override
  public void onInterstitialClicked(@NonNull Interstitial interstitial) {
    if (isDestroyed()) {
      return;
    }

    MaxAdsLog.d(TAG, "Interstitial clicked for ad unit id: " + mAd.getAdUnitId());
    mAd.trackClickUrls();
    if (mListener != null) {
      mListener.onInterstitialClicked(this);
    }
  }

  @Override
  public void onInterstitialDismissed(@NonNull Interstitial interstitial) {
    if (!mInterstitialState.transitionState(InterstitialState.State.IDLE)) {
      return;
    }

    MaxAdsLog.d(TAG, "Interstitial dismissed for ad unit id: " + mAd.getAdUnitId());
    if (mListener != null) {
      mListener.onInterstitialDismissed(this);
    }
  }

  @Override
  public void onInterstitialDidExpire(@NonNull Interstitial interstitial) {
    if (!mInterstitialState.transitionState(InterstitialState.State.IDLE)) {
      return;
    }

    MaxAdsLog.d(TAG, "Interstitial expired for ad unit id: " + mAd.getAdUnitId());
    if (mListener != null) {
      mListener.onInterstitialDidExpire(this);
    }
  }

  @Override
  public void onInterstitialError(@NonNull Interstitial interstitial) {
    if (!mInterstitialState.transitionState(InterstitialState.State.IDLE)) {
      return;
    }

    final String errorMessage = "Interstitial error for ad unit id: " + mAd.getAdUnitId();
    MaxAdsLog.d(TAG, errorMessage);
    mAd.trackError(errorMessage);
    if (mListener != null) {
      mListener.onInterstitialError(this);
    }
  }

  // InterstitialLoadTimer.Listener

  @Override
  public void onTimerComplete() {
    onInterstitialFailedToLoad(mInterstitial);
  }
}
