package io.maxads.ads.base.util;

import android.content.Context;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import android.util.TypedValue;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;

import static android.Manifest.permission.ACCESS_NETWORK_STATE;
import static android.Manifest.permission.INTERNET;

public class Util {
  // When converting float to int, add this constant to ensure it rounds to the next highest int
  private static final float ROUND_UP = 0.5f;

  public static boolean isNetworkAvailable(@Nullable final Context context) {
    if (context == null) {
      return false;
    }

    if (!Permissions.hasPermission(context, INTERNET)) {
      return false;
    }

    if (!Permissions.hasPermission(context, ACCESS_NETWORK_STATE)) {
      return true;
    }

    final ConnectivityManager connectionManager =
      (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
    if (connectionManager == null) {
      return false;
    }

    final NetworkInfo networkInfo = connectionManager.getActiveNetworkInfo();
    return networkInfo != null && networkInfo.isConnectedOrConnecting();
  }

  @Nullable
  public static String inputStreamToString(@Nullable InputStream inputStream) {
    if (inputStream == null) {
      return null;
    }

    final BufferedReader bufferedReader = new BufferedReader(new InputStreamReader(inputStream));
    String line;
    final StringBuilder stringBuilder = new StringBuilder();
    try {
      while ((line = bufferedReader.readLine()) != null) {
        stringBuilder.append(line);
      }
      bufferedReader.close();
    } catch (IOException ignore) {
      return null;
    }

    return stringBuilder.toString();
  }

  public static float dipsToPixels(float valueDp, @NonNull Context context) {
    return TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, valueDp, context.getResources().getDisplayMetrics());
  }

  public static int dipsToIntPixels(float valueDp, @NonNull Context context) {
    return (int) (dipsToPixels(valueDp, context) + ROUND_UP);
  }

  public static int pixelsToIntDips(float valuePx, @NonNull Context context) {
    return (int) (valuePx / context.getResources().getDisplayMetrics().density + ROUND_UP);
  }

  public static boolean enumContainsValue(@NonNull Class<? extends Enum> enumClass, @Nullable String value) {
    if (value == null) {
      return false;
    }

    try {
      Enum.valueOf(enumClass, value);
      return true;
    } catch (Exception e) {
      return false;
    }
  }

  @NonNull
  public static String combineKeywords(@Nullable String maxKeywords, @Nullable String pubKeywords) {
    final boolean maxKeywordsEmpty = TextUtils.isEmpty(maxKeywords);
    final boolean pubKeywordsEmpty = TextUtils.isEmpty(pubKeywords);

    if (!maxKeywordsEmpty && !pubKeywordsEmpty) {
      return maxKeywords + "," + pubKeywords;
    } else if (!maxKeywordsEmpty) {
      return maxKeywords;
    } else if (!pubKeywordsEmpty) {
      return pubKeywords;
    }

    return "";
  }

  public static boolean equals(@Nullable Object a, @Nullable Object b) {
    return (a == b) || (a != null && a.equals(b));
  }

  @SuppressWarnings("UseCompareMethod")
  public static int compare(int x, int y) {
    return (x < y) ? -1 : ((x == y) ? 0 : 1);
  }
}
