package io.maxads.ads.base.location;

import android.content.Context;
import android.location.Location;
import android.location.LocationManager;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;

import com.google.android.gms.common.ConnectionResult;
import com.google.android.gms.common.GoogleApiAvailability;
import com.jenzz.appstate.AppState;

import io.maxads.ads.base.AppStateManager;
import io.maxads.ads.base.util.Optional;
import io.reactivex.Observable;

public class MaxLocationManager implements LocationProvider, AppStateManager.Listener {
  @NonNull private static final EmptyLocationProvider EMPTY_LOCATION_PROVIDER = new EmptyLocationProvider();

  @NonNull private final Context mContext;
  @NonNull private LocationProvider mLocationProvider;
  @NonNull private final GoogleApiAvailability mGoogleApiAvailability;
  @Nullable private final LocationManager mLocationManager;
  @NonNull private LocationTrackingStatus mLocationTrackingStatus;
  private boolean mLocationTrackingEnabled;

  public MaxLocationManager(@NonNull Context context) {
    this(context.getApplicationContext(), new EmptyLocationProvider(), GoogleApiAvailability.getInstance(),
      (LocationManager) context.getSystemService(Context.LOCATION_SERVICE));
  }

  @VisibleForTesting
  MaxLocationManager(@NonNull Context context,
                     @NonNull LocationProvider locationProvider,
                     @NonNull GoogleApiAvailability googleApiAvailability,
                     @Nullable LocationManager locationManager) {
    mContext = context;
    mLocationProvider = locationProvider;
    mGoogleApiAvailability = googleApiAvailability;
    mLocationManager = locationManager;
    mLocationTrackingStatus = LocationTrackingStatus.DISABLED;
  }

  @NonNull
  public LocationTrackingStatus getLocationTrackingStatus() {
    return mLocationTrackingStatus;
  }

  public void enableLocationTracking() {
    if (mLocationTrackingEnabled) {
      return;
    }

    mLocationProvider = initializeLocationProvider(mContext);
    startLocationUpdates();
    mLocationTrackingEnabled = true;
  }

  public void disableLocationTracking() {
    if (!mLocationTrackingEnabled) {
      return;
    }

    stopLocationUpdates();
    mLocationProvider = EMPTY_LOCATION_PROVIDER;
    mLocationTrackingEnabled = false;
  }

  @Override
  public void startLocationUpdates() {
    mLocationProvider.startLocationUpdates();
  }

  @Override
  public void stopLocationUpdates() {
    mLocationProvider.stopLocationUpdates();
  }

  @NonNull
  @Override
  public Observable<Optional<Location>> getLastLocationAndStartUpdates() {
    // Need to check and update this each time we get a location since permissions may have changed
    mLocationTrackingStatus = updateLocationTrackingStatus();
    return mLocationProvider.getLastLocationAndStartUpdates();
  }

  @Override
  public boolean hasPermission() {
    return mLocationProvider.hasPermission();
  }

  @NonNull
  private LocationProvider initializeLocationProvider(@NonNull Context context) {
    return initializeLocationProvider(context, mGoogleApiAvailability, mLocationManager);
  }

  @NonNull
  @VisibleForTesting
  LocationProvider initializeLocationProvider(@NonNull Context context,
                                              @NonNull GoogleApiAvailability googleApiAvailability,
                                              @Nullable LocationManager locationManager) {
    final LocationProvider locationProvider;
    if (googleApiAvailability.isGooglePlayServicesAvailable(context) == ConnectionResult.SUCCESS) {
      locationProvider = new FusedLocationProvider(context);
    } else if (locationManager != null) {
      locationProvider = new LocationManagerProvider(context, locationManager);
    } else {
      locationProvider = EMPTY_LOCATION_PROVIDER;
    }
    return locationProvider;
  }

  @NonNull
  private LocationTrackingStatus updateLocationTrackingStatus() {
    return updateLocationTrackingStatus(mLocationTrackingEnabled, mLocationProvider.hasPermission());
  }

  @NonNull
  @VisibleForTesting
  LocationTrackingStatus updateLocationTrackingStatus(boolean locationTrackingEnabled, boolean providerHasPermission) {
    if (!locationTrackingEnabled) {
      return LocationTrackingStatus.DISABLED;
    } else if (providerHasPermission) {
      return LocationTrackingStatus.ENABLED;
    } else {
      return LocationTrackingStatus.UNAUTHORIZED;
    }
  }

  @Override
  public void onAppStateChanged(AppState appState) {
    if (appState == AppState.BACKGROUND) {
      stopLocationUpdates();
    } else if (appState == AppState.FOREGROUND) {
      startLocationUpdates();
    }
  }
}
