package io.maxads.ads.base.location;

import android.Manifest;
import android.annotation.SuppressLint;
import android.content.Context;
import android.location.Location;
import android.location.LocationListener;
import android.location.LocationManager;
import android.os.Bundle;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import java.util.concurrent.TimeUnit;

import io.maxads.ads.base.util.MaxAdsLog;
import io.maxads.ads.base.util.Optional;
import io.maxads.ads.base.util.Permissions;
import io.reactivex.Observable;

/**
 * References for implementation:
 * https://developer.android.com/guide/topics/location/strategies.html
 * https://www.androidhive.info/2012/07/android-gps-location-manager-tutorial/
 */
@SuppressLint("MissingPermission")
public class LocationManagerProvider implements LocationProvider, LocationListener {
  @NonNull private static final String TAG = LocationManagerProvider.class.getSimpleName();

  private static final long TWO_MINUTES = TimeUnit.MINUTES.toSeconds(2);
  private static final long INTERVAL_MS = TimeUnit.MINUTES.toMillis(1);
  private static final float DISTANCE_M = 10f;

  @NonNull private final Context mContext;
  @NonNull private final LocationManager mLocationManager;
  @Nullable private Location mLastLocation;
  private boolean mLocationUpdatesStarted;

  public LocationManagerProvider(@NonNull Context context, @NonNull LocationManager locationManager) {
    mContext = context.getApplicationContext();
    mLocationManager = locationManager;
  }

  @Override
  public void startLocationUpdates() {
    if (mLocationUpdatesStarted) {
      return;
    }

    if (hasFinePermission() && mLocationManager.isProviderEnabled(LocationManager.GPS_PROVIDER)) {
      mLocationManager.requestLocationUpdates(LocationManager.GPS_PROVIDER, INTERVAL_MS, DISTANCE_M, this);
      mLocationUpdatesStarted = true;
    }

    if (hasCoarsePermission() && mLocationManager.isProviderEnabled(LocationManager.NETWORK_PROVIDER)) {
      mLocationManager.requestLocationUpdates(LocationManager.NETWORK_PROVIDER, INTERVAL_MS, DISTANCE_M, this);
      mLocationUpdatesStarted = true;
    }
  }

  @Override
  public void stopLocationUpdates() {
    mLocationManager.removeUpdates(this);
    mLocationUpdatesStarted = false;
  }

  @NonNull
  @Override
  public Observable<Optional<Location>> getLastLocationAndStartUpdates() {
    if (!hasPermission()) {
      return Observable.just(Optional.<Location>empty());
    }

    startLocationUpdates();

    if (mLastLocation != null) {
      return Observable.just(Optional.of(mLastLocation));
    }

    if (hasFinePermission() && mLocationManager.isProviderEnabled(LocationManager.GPS_PROVIDER)) {
      try {
        mLastLocation = mLocationManager.getLastKnownLocation(LocationManager.GPS_PROVIDER);
        return Observable.just(Optional.ofNullable(mLastLocation));
      } catch (Exception e) {
        MaxAdsLog.d(TAG, "Failed to get last known GPS location", e);
      }
    }

    if (hasCoarsePermission() && mLocationManager.isProviderEnabled(LocationManager.NETWORK_PROVIDER)) {
      try {
        mLastLocation = mLocationManager.getLastKnownLocation(LocationManager.NETWORK_PROVIDER);
        return Observable.just(Optional.ofNullable(mLastLocation));
      } catch (Exception e) {
        MaxAdsLog.d(TAG, "Failed to get last known network location", e);
      }
    }

    return Observable.just(Optional.<Location>empty());
  }

  private boolean hasCoarsePermission() {
    return Permissions.hasPermission(mContext, Manifest.permission.ACCESS_COARSE_LOCATION);
  }

  private boolean hasFinePermission() {
    return Permissions.hasPermission(mContext, Manifest.permission.ACCESS_FINE_LOCATION);
  }

  @Override
  public boolean hasPermission() {
    return hasCoarsePermission() || hasFinePermission();
  }

  // LocationListener

  @Override
  public void onLocationChanged(Location location) {
    mLastLocation = isBetterLocation(location, mLastLocation) ? location : mLastLocation;
  }

  @Override
  public void onStatusChanged(String provider, int status, Bundle extras) {
  }

  @Override
  public void onProviderEnabled(String provider) {
  }

  @Override
  public void onProviderDisabled(String provider) {
  }

  /**
   * Borrowed from: https://developer.android.com/guide/topics/location/strategies.html
   * Determines whether one Location reading is better than the current Location fix
   * @param location  The new Location to evaluate
   * @param currentBestLocation  The current Location fix, to which you want to compare the new one
   */
  private boolean isBetterLocation(@NonNull Location location, @Nullable Location currentBestLocation) {
    if (currentBestLocation == null) {
      // A new location is always better than no location
      return true;
    }

    // Check whether the new location fix is newer or older
    long timeDelta = location.getTime() - currentBestLocation.getTime();
    boolean isSignificantlyNewer = timeDelta > TWO_MINUTES;
    boolean isSignificantlyOlder = timeDelta < -TWO_MINUTES;
    boolean isNewer = timeDelta > 0;

    // If it's been more than two minutes since the current location, use the new location
    // because the user has likely moved
    if (isSignificantlyNewer) {
      return true;
      // If the new location is more than two minutes older, it must be worse
    } else if (isSignificantlyOlder) {
      return false;
    }

    // Check whether the new location fix is more or less accurate
    int accuracyDelta = (int) (location.getAccuracy() - currentBestLocation.getAccuracy());
    boolean isLessAccurate = accuracyDelta > 0;
    boolean isMoreAccurate = accuracyDelta < 0;
    boolean isSignificantlyLessAccurate = accuracyDelta > 200;

    // Check if the old and new location are from the same provider
    boolean isFromSameProvider = isSameProvider(location.getProvider(),
      currentBestLocation.getProvider());

    // Determine location quality using a combination of timeliness and accuracy
    if (isMoreAccurate) {
      return true;
    } else if (isNewer && !isLessAccurate) {
      return true;
    } else if (isNewer && !isSignificantlyLessAccurate && isFromSameProvider) {
      return true;
    }
    return false;
  }

  /**
   * Borrowed from: https://developer.android.com/guide/topics/location/strategies.html
   * Checks whether two providers are the same
   */
  private boolean isSameProvider(@Nullable String provider1, @Nullable String provider2) {
    if (provider1 == null) {
      return provider2 == null;
    }
    return provider1.equals(provider2);
  }
}
