package io.maxads.ads.base.location;

import android.Manifest;
import android.annotation.SuppressLint;
import android.content.Context;
import android.location.Location;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;

import com.google.android.gms.location.FusedLocationProviderClient;
import com.google.android.gms.location.LocationCallback;
import com.google.android.gms.location.LocationRequest;
import com.google.android.gms.location.LocationResult;
import com.google.android.gms.location.LocationServices;
import com.google.android.gms.tasks.OnFailureListener;
import com.google.android.gms.tasks.OnSuccessListener;

import java.util.concurrent.TimeUnit;

import io.maxads.ads.base.util.MaxAdsLog;
import io.maxads.ads.base.util.Optional;
import io.maxads.ads.base.util.Permissions;
import io.reactivex.Observable;
import io.reactivex.ObservableEmitter;
import io.reactivex.ObservableOnSubscribe;

/**
 * References for implementation:
 * https://developer.android.com/training/location/index.html
 * https://developers.google.com/android/reference/com/google/android/gms/location/FusedLocationProviderClient.html
 * https://developers.google.com/android/reference/com/google/android/gms/location/LocationRequest
 */
@SuppressLint("MissingPermission")
public class FusedLocationProvider extends LocationCallback implements LocationProvider {
  @NonNull private static final String TAG = FusedLocationProvider.class.getSimpleName();

  private static final long INTERVAL_MS = TimeUnit.MINUTES.toMillis(60);
  private static final long FASTEST_INTERVAL_MS = TimeUnit.MINUTES.toMillis(1);

  @NonNull private final Context mContext;
  @NonNull private final FusedLocationProviderClient mFusedLocationProviderClient;
  @NonNull private final LocationRequest mLocationRequest;
  @Nullable private Location mLastLocation;
  private boolean mLocationUpdatesStarted;

  public FusedLocationProvider(@NonNull Context context) {
    this(context.getApplicationContext(), LocationServices.getFusedLocationProviderClient(context),
      new LocationRequest());
  }

  @VisibleForTesting
  FusedLocationProvider(@NonNull Context context,
                        @NonNull FusedLocationProviderClient fusedLocationProviderClient,
                        @NonNull LocationRequest locationRequest) {
    mContext = context;
    mFusedLocationProviderClient = fusedLocationProviderClient;
    mLocationRequest = locationRequest;
    mLocationRequest.setInterval(INTERVAL_MS)
      .setFastestInterval(FASTEST_INTERVAL_MS);
  }

  @Override
  public void startLocationUpdates() {
    if (mLocationUpdatesStarted || !hasPermission()) {
      return;
    }

    if (hasFinePermission()) {
      mLocationRequest.setPriority(LocationRequest.PRIORITY_HIGH_ACCURACY);
    } else if (hasCoarsePermission()) {
      mLocationRequest.setPriority(LocationRequest.PRIORITY_BALANCED_POWER_ACCURACY);
    }

    mFusedLocationProviderClient.requestLocationUpdates(mLocationRequest, this, null);
    mLocationUpdatesStarted = true;
  }

  @Override
  public void stopLocationUpdates() {
    mFusedLocationProviderClient.removeLocationUpdates(this);
    mLocationUpdatesStarted = false;
  }

  @NonNull
  @Override
  public Observable<Optional<Location>> getLastLocationAndStartUpdates() {
    if (!hasPermission()) {
      return Observable.just(Optional.<Location>empty());
    }

    startLocationUpdates();

    if (mLastLocation != null) {
      return Observable.just(Optional.of(mLastLocation));
    }

    return Observable.create(new ObservableOnSubscribe<Optional<Location>>() {
      @Override
      public void subscribe(final ObservableEmitter<Optional<Location>> observableEmitter) {
        try {
          mFusedLocationProviderClient.getLastLocation()
            .addOnSuccessListener(new OnSuccessListener<Location>() {
              @Override
              public void onSuccess(@Nullable Location location) {
                mLastLocation = location;
                observableEmitter.onNext(Optional.ofNullable(location));
                observableEmitter.onComplete();
              }
            })
            .addOnFailureListener(new OnFailureListener() {
              @Override
              public void onFailure(@NonNull Exception e) {
                observableEmitter.onNext(Optional.<Location>empty());
                observableEmitter.onComplete();
              }
            });
        } catch (Exception e) {
          // Just in case we forgot about other random exceptions
          MaxAdsLog.d(TAG, "Failed to get last known location", e);
          observableEmitter.onNext(Optional.<Location>empty());
          observableEmitter.onComplete();
        }
      }
    });
  }

  private boolean hasCoarsePermission() {
    return Permissions.hasPermission(mContext, Manifest.permission.ACCESS_COARSE_LOCATION);
  }

  private boolean hasFinePermission() {
    return Permissions.hasPermission(mContext, Manifest.permission.ACCESS_FINE_LOCATION);
  }

  @Override
  public boolean hasPermission() {
    return hasCoarsePermission() || hasFinePermission();
  }

  @Override
  public void onLocationResult(LocationResult locationResult) {
    super.onLocationResult(locationResult);
    if (locationResult != null) {
      mLastLocation = locationResult.getLastLocation();
    }
  }
}
