package io.maxads.ads.base.cache;

import android.annotation.SuppressLint;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;
import android.util.LruCache;
import android.webkit.WebView;

import com.jenzz.appstate.AppState;

import java.lang.ref.WeakReference;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import io.maxads.ads.base.AppStateManager;
import io.maxads.ads.base.util.Helpers;
import io.maxads.ads.interstitial.Interstitial;
import io.maxads.ads.interstitial.MaxInterstitial;
import io.reactivex.disposables.Disposable;
import io.reactivex.functions.Consumer;
import io.reactivex.schedulers.Schedulers;

/**
 * Cache used for storing pre-rendered web views for use in interstitials.
 * The web view should be pre-rendered before inserting in the cache and then retrieved in the interstitial's
 * activity for display.
 * This allows the publisher to only call {@link MaxInterstitial#show()} when the ad is ready to be displayed.
 */
public class WebViewCache implements AppStateManager.Listener {
  @NonNull private static final String TAG = WebViewCache.class.getSimpleName();

  public static class WebViewLruCache extends LruCache<Long, Data> {
    public WebViewLruCache(int maxSize) {
      super(maxSize);
    }

    @Override
    protected void entryRemoved(boolean evicted, Long key, Data oldValue, Data newValue) {
      super.entryRemoved(evicted, key, oldValue, newValue);

      if (evicted) {
        oldValue.getWebView().destroy();
      }
    }
  }

  public static class Data {
    @NonNull private final WebView mWebView;
    @NonNull private final WeakReference<Interstitial> mInterstitial;

    public Data(@NonNull WebView webView, @NonNull Interstitial interstitial) {
      mWebView = webView;
      mInterstitial = new WeakReference<>(interstitial);
    }

    @NonNull
    public WebView getWebView() {
      return mWebView;
    }

    @NonNull
    public WeakReference<Interstitial> getInterstitial() {
      return mInterstitial;
    }
  }

  private static final int MAX_SIZE = 50;
  private static final long TRIM_INTERVAL_MS = TimeUnit.MINUTES.toMillis(10);
  @NonNull private final WebViewLruCache mWebViewLruCache;
  @NonNull private final Helpers.IntervalFactory mIntervalFactory;
  @Nullable private Disposable mDisposable;

  @SuppressLint("UseSparseArrays")
  public WebViewCache() {
    this(new WebViewLruCache(MAX_SIZE), new Helpers.IntervalFactory());
  }

  @VisibleForTesting
  WebViewCache(@NonNull WebViewLruCache webViewLruCache, @NonNull Helpers.IntervalFactory intervalFactory) {
    mWebViewLruCache = webViewLruCache;
    mIntervalFactory = intervalFactory;
  }

  public void put(long broadcastId, @NonNull Data data) {
    trimWebViewCache();
    mWebViewLruCache.put(broadcastId, data);
    startTrimInterval();
  }

  @Nullable
  public Data remove(@Nullable Long broadcastId) {
    if (broadcastId == null) {
      return null;
    }

    final Data data = mWebViewLruCache.remove(broadcastId);

    if (isEmpty()) {
      stopTrimInterval();
    }

    return data;
  }

  private boolean isEmpty() {
    return mWebViewLruCache.size() <= 0;
  }

  @VisibleForTesting
  void trimWebViewCache() {
    if (isEmpty()) {
      return;
    }

    final Iterator<Map.Entry<Long, Data>> iterator = mWebViewLruCache.snapshot().entrySet().iterator();
    while (iterator.hasNext()) {
      final Map.Entry<Long, Data> entry = iterator.next();

      // If the interstitial is garbage collected then it means that we can safely assume that the webview
      // has already been used or it was not used and failed to be removed from the cache
      final Data value = entry.getValue();
      if (value.getInterstitial().get() == null) {
        value.getWebView().destroy();
        mWebViewLruCache.remove(entry.getKey());
      }
    }
  }

  private void startTrimInterval() {
    doStartTrimInterval(mDisposable);
  }

  @VisibleForTesting
  void doStartTrimInterval(@Nullable Disposable disposable) {
    if (isEmpty() || (disposable != null && !disposable.isDisposed())) {
      return;
    }

    mDisposable = mIntervalFactory.createInterval(TRIM_INTERVAL_MS, TRIM_INTERVAL_MS, TimeUnit.MILLISECONDS)
      .observeOn(Schedulers.computation())
      .subscribe(new Consumer<Long>() {
        @Override
        public void accept(Long aLong) {
          trimWebViewCache();
        }
      });
  }

  private void stopTrimInterval() {
    doStopTrimInterval(mDisposable);
  }

  @VisibleForTesting
  void doStopTrimInterval(@Nullable Disposable disposable) {
    if (disposable != null) {
      disposable.dispose();
    }
  }

  @Override
  public void onAppStateChanged(AppState appState) {
    if (appState == AppState.BACKGROUND) {
      stopTrimInterval();
    } else if (appState == AppState.FOREGROUND) {
      startTrimInterval();
    }
  }
}
