package io.maxads.ads.base.cache;

import android.content.Context;
import android.os.StatFs;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;
import android.support.annotation.WorkerThread;
import android.text.TextUtils;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.util.concurrent.Callable;

import io.maxads.ads.base.util.MaxAdsLog;
import io.reactivex.Observable;
import io.reactivex.ObservableSource;
import io.reactivex.schedulers.Schedulers;
import okhttp3.internal.cache.DiskLruCache;
import okhttp3.internal.io.FileSystem;
import okio.BufferedSink;
import okio.BufferedSource;
import okio.ByteString;
import okio.Okio;

// Implementation references:
// https://www.programcreek.com/java-api-examples/index.php?api=com.jakewharton.disklrucache.DiskLruCache
public class MaxDiskLruCacheImpl implements MaxDiskLruCache {
  @NonNull private static final String TAG = MaxDiskLruCacheImpl.class.getSimpleName();

  @VisibleForTesting
  public static class DiskSpaceHelper {
    @NonNull public static final DiskSpaceHelper DEFAULT = new DiskSpaceHelper();
    public long getAvailableBytes(@NonNull StatFs statFs) {
      return ((long) statFs.getAvailableBlocks()) * statFs.getBlockSize();
    }
  }

  @NonNull private static final String CACHE_NAME = "max-cache";
  private static final int DISK_CACHE_INDEX = 0;

  private static final int MIN_DISK_CACHE_SIZE_BYTES = 30 * 1024 * 1024;  // 30 MB
  private static final int MAX_DISK_CACHE_SIZE_BYTES = 100 * 1024 * 1024; // 100 MB
  private static final int MAX_VIDEO_SIZE_BYTES = 3 * 1024 * 1024; // 3 MB

  private static final int APP_VERSION = 1;
  // The number of values per cache entry. Must be positive.
  private static final int VALUE_COUNT = 1;

  @NonNull private final DiskLruCache mDiskLruCache;

  @Nullable
  public static MaxDiskLruCacheImpl initialize(@NonNull Context context) {
    final File cacheDir = getDiskCacheDirectory(context);
    if (cacheDir == null) {
      return null;
    }

    final long diskCacheSizeBytes = getDiskCacheSizeBytes(cacheDir, MIN_DISK_CACHE_SIZE_BYTES);
    return new MaxDiskLruCacheImpl(
      DiskLruCache.create(FileSystem.SYSTEM, cacheDir, APP_VERSION, VALUE_COUNT, diskCacheSizeBytes));
  }

  @VisibleForTesting
  @Nullable
  static File getDiskCacheDirectory(@NonNull final Context context) {
    final File cacheDir = context.getCacheDir();
    if (cacheDir == null) {
      MaxAdsLog.w(TAG, "Unable to get disk cache directory");
      return null;
    }

    final String cachePath = cacheDir.getPath();
    return new File(cachePath + File.separator + CACHE_NAME);
  }

  private static long getDiskCacheSizeBytes(@NonNull File file, long minSize) {
    return getDiskCacheSizeBytes(file, minSize, DiskSpaceHelper.DEFAULT);
  }

  // Implementation references:
  // https://www.programcreek.com/java-api-examples/index.php?class=android.os.StatFs&method=getAvailableBytes
  @VisibleForTesting
  static long getDiskCacheSizeBytes(@NonNull File file, long minSize, @NonNull DiskSpaceHelper diskSpaceHelper) {
    long size = minSize;
    try {
      // If we do not have storage permissions this call will throw an exception
      // From testing it seems like the rest of the disk cache can still function correctly without this permission
      final StatFs statFs = new StatFs(file.getAbsolutePath());
      final long availableBytes = diskSpaceHelper.getAvailableBytes(statFs);

      // Only use 2% of available disk space
      size = (long) (availableBytes * 0.02);
    } catch (Exception e) {
      MaxAdsLog.w(TAG, "Unable to calculate 2% of available disk space, defaulting to minimum");
    }

    // Bound inside min/max size for disk cache.
    return Math.max(Math.min(size, MAX_DISK_CACHE_SIZE_BYTES), MIN_DISK_CACHE_SIZE_BYTES);
  }

  public MaxDiskLruCacheImpl(@NonNull DiskLruCache diskLruCache) {
    mDiskLruCache = diskLruCache;
  }

  @NonNull
  @Override
  public String getFilePath(@Nullable String key) {
    if (TextUtils.isEmpty(key)) {
      return "";
    }

    return mDiskLruCache.getDirectory() + File.separator + key(key) + "." + DISK_CACHE_INDEX;
  }

  @WorkerThread
  @Override
  public boolean put(@NonNull String key, @NonNull BufferedSource bufferedSource) {
    DiskLruCache.Editor editor = null;
    try {
      editor = mDiskLruCache.edit(key(key));
      if (editor == null) {
        // another edit is in progress
        return false;
      }

      writeToSink(editor, bufferedSource);
    } catch (Exception e) {
      MaxAdsLog.w(TAG, "Failed to put key: " + key, e);
      abortQuietly(editor);
      return false;
    } finally {
      closeQuietly(bufferedSource);
    }
    return true;
  }

  @NonNull
  @Override
  public Observable<Void> putAsync(@Nullable final String key,
                                   @NonNull final BufferedSource bufferedSource,
                                   long sizeBytes) {
    if (TextUtils.isEmpty(key)) {
      closeQuietly(bufferedSource);
      return Observable.error(new Exception("Key is null or empty"));
    }

    if (sizeBytes > Math.min(MAX_VIDEO_SIZE_BYTES, mDiskLruCache.getMaxSize())) {
      closeQuietly(bufferedSource);
      return Observable.error(new Exception("Value too large for key: " + key));
    }

    return Observable.defer(new Callable<ObservableSource<? extends Void>>() {
      @Override
      public ObservableSource<? extends Void> call() {
        return put(key, bufferedSource)
          ? Observable.<Void>empty()
          : Observable.<Void>error(new Exception("Failed to put key to MaxDiskLruCache: " + key));
      }
    }).subscribeOn(Schedulers.io());
  }

  @WorkerThread
  private void writeToSink(@NonNull DiskLruCache.Editor editor, @NonNull BufferedSource bufferedSource)
    throws IOException {
    final BufferedSink sink = Okio.buffer(editor.newSink(DISK_CACHE_INDEX));
    try {
      bufferedSource.readAll(sink);
      sink.flush();
    } finally {
      sink.close();
    }

    mDiskLruCache.flush();
    editor.commit();
  }

  @Override
  public boolean containsKey(@Nullable String key) {
    if (TextUtils.isEmpty(key)) {
      return false;
    }

    try {
      return mDiskLruCache.get(key(key)) != null;
    } catch (Exception e) {
      return false;
    }
  }

  @NonNull
  private String key(@NonNull String key) {
    // Borrowed from okhttp3.Cache#key
    try {
      return ByteString.encodeUtf8(key).md5().hex();
    } catch (Exception e) {
      return String.valueOf(key.hashCode());
    }
  }

  private void abortQuietly(@Nullable DiskLruCache.Editor editor) {
    // Give up because the cache cannot be written.
    if (editor == null) {
      return;
    }

    try {
      editor.abort();
    } catch (IOException ignored) {
    }
  }

  private void closeQuietly(@Nullable Closeable closeable) {
    if (closeable == null) {
      return;
    }

    try {
      closeable.close();
    } catch (IOException ignored) {
    }
  }
}
