package io.maxads.ads.base.api;

import android.annotation.SuppressLint;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;

import com.jenzz.appstate.AppState;

import io.maxads.ads.base.AppStateManager;
import io.maxads.ads.base.MaxAds;
import io.maxads.ads.base.RefreshTimer;
import io.maxads.ads.base.cache.AdCache;
import io.maxads.ads.base.model.Ad;
import io.maxads.ads.base.util.Checks;
import io.maxads.ads.base.util.Helpers;
import io.maxads.ads.base.util.MaxAdsLog;
import io.reactivex.ObservableSource;
import io.reactivex.disposables.Disposable;
import io.reactivex.functions.Action;
import io.reactivex.functions.Consumer;
import io.reactivex.functions.Function;

public class MaxRequestManager implements RefreshTimer.Listener, AppStateManager.Listener {
  @NonNull private static final String TAG = MaxRequestManager.class.getSimpleName();

  public interface RequestListener {
    void onRequestSuccess(@NonNull Ad ad);
    void onRequestFail(@NonNull Throwable throwable);
  }

  @NonNull private final ApiClient mApiClient;
  @NonNull private final AdCache mAdCache;
  @NonNull private final AdRequestFactory mAdRequestFactory;
  @NonNull private final RefreshTimer mRefreshTimer;
  @NonNull private final AppStateManager mAppStateManager;
  @NonNull private final Helpers.InitializationHelper mInitializationHelper;
  @Nullable private String mAdUnitId;
  @Nullable private AdRequestParameters mAdRequestParameters;
  @Nullable private Disposable mDisposable;
  @Nullable private RequestListener mRequestListener;
  private boolean mIsDestroyed;

  public MaxRequestManager() {
    this(MaxAds.getApiClient(), MaxAds.getAdCache(), new AdRequestFactory(), new RefreshTimer(),
      MaxAds.getAppStateManager(), new Helpers.InitializationHelper());
  }

  @VisibleForTesting
  MaxRequestManager(@NonNull ApiClient apiClient,
                    @NonNull AdCache adCache,
                    @NonNull AdRequestFactory adRequestFactory,
                    @NonNull RefreshTimer refreshTimer,
                    @NonNull AppStateManager appStateManager,
                    @NonNull Helpers.InitializationHelper initializationHelper) {
    mApiClient = apiClient;
    mAdCache = adCache;
    mAdRequestFactory = adRequestFactory;
    mRefreshTimer = refreshTimer;
    mRefreshTimer.setListener(this);
    mAppStateManager = appStateManager;
    mAppStateManager.addListener(this);
    mInitializationHelper = initializationHelper;
  }

  public void setRequestListener(@Nullable RequestListener requestListener) {
    mRequestListener = requestListener;
  }

  public void requestAd(@NonNull String adUnitId) {
    requestAd(adUnitId, AdRequestParameters.EMPTY_PARAMETERS);
  }

  @SuppressLint("CheckResult")
  public void requestAd(@NonNull final String adUnitId, @NonNull AdRequestParameters adRequestParameters) {
    if (!mInitializationHelper.isInitialized()) {
      return;
    }

    if (!Checks.NoThrow.checkNotNull(adUnitId, "adUnitId cannot be null")) {
      return;
    }

    if (!Checks.NoThrow.checkNotNull(adRequestParameters, "adRequestParameters cannot be null")) {
      return;
    }

    if (!Checks.NoThrow.checkArgument(!mIsDestroyed, TAG + " has been destroyed")) {
      return;
    }

    MaxAdsLog.d(TAG, "Requesting ad for ad unit id: " + adUnitId);

    mAdUnitId = adUnitId;
    mAdRequestParameters = adRequestParameters;
    mDisposable = mAdRequestFactory.createAdRequest(adUnitId, adRequestParameters)
      .flatMap(new Function<AdRequest, ObservableSource<Ad>>() {
        @Override
        public ObservableSource<Ad> apply(AdRequest adRequest) {
          return mApiClient.getAd(adRequest);
        }
      })
      .subscribe(
        new Consumer<Ad>() {
          @Override
          public void accept(Ad ad) {
            MaxAdsLog.d(TAG, "Received ad response for ad unit id: " + ad.getAdUnitId());
            mAdCache.put(ad);
            if (mRequestListener != null) {
              mRequestListener.onRequestSuccess(ad);
            }
          }
        }, new Consumer<Throwable>() {
          @Override
          public void accept(Throwable throwable) {
            disposeDisposable();

            MaxAdsLog.w(TAG, "Failed to receive ad response for ad unit id: " + adUnitId
              + ", message: " + throwable.getMessage());
            if (mRequestListener != null) {
              mRequestListener.onRequestFail(throwable);
            }
          }
        },
        new Action() {
          @Override
          public void run() {
            disposeDisposable();
          }
        }
      );
  }

  public boolean isLoading() {
    return mDisposable != null && !mDisposable.isDisposed();
  }

  public void startRefreshTimer(long delaySeconds) {
    if (!Checks.NoThrow.checkArgument(!mIsDestroyed, TAG + " has been destroyed")) {
      return;
    }

    delaySeconds = delaySeconds > 0 ? delaySeconds : RefreshTimer.DEFAULT_REFRESH_TIME_SECONDS;
    mRefreshTimer.start(delaySeconds);
  }

  public void stopRefreshTimer() {
    mRefreshTimer.stop();
  }

  public void destroy() {
    mRefreshTimer.destroy();
    mAppStateManager.removeListener(this);
    disposeDisposable();
    mRequestListener = null;
    mIsDestroyed = true;
  }

  private void disposeDisposable() {
    if (mDisposable != null) {
      mDisposable.dispose();
    }
  }

  @Override
  public void onTimerComplete() {
    requestAd(mAdUnitId, mAdRequestParameters);
  }

  @Override
  public void onAppStateChanged(AppState appState) {
    if (mIsDestroyed) {
      return;
    }

    if (appState == AppState.FOREGROUND) {
      mRefreshTimer.resume();
    } else {
      mRefreshTimer.pause();
    }
  }
}
