package io.maxads.ads.base.api;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;

import java.util.Locale;
import java.util.concurrent.TimeUnit;

import io.maxads.ads.base.util.Checks;
import io.maxads.ads.base.util.MaxAdsLog;
import io.reactivex.Observable;
import io.reactivex.ObservableSource;
import io.reactivex.functions.BiFunction;
import io.reactivex.functions.Function;

public class ExponentialBackoff implements Function<Observable<? extends Throwable>, Observable<Long>> {
  @NonNull private static final String TAG = ExponentialBackoff.class.getSimpleName();

  @Nullable private final String mUrl;
  @NonNull private final Jitter mJitter;
  private final long mDelay;
  private final long mMaxDelay;
  @NonNull private final TimeUnit mTimeUnit;
  private final int mMaxRetries;

  /**
   * Exponential backoff that respects the equation: min(mDelay * mMaxRetries^2 * mJitter, mMaxDelay * mJitter)
   *
   * @param jitter     Adds a small bit of variation to each backoff interval calculated
   * @param delay      The base delay to wait before trying again
   * @param maxDelay   The max delay to wait before trying again
   * @param timeUnit   The time unit
   * @param maxRetries The max number of retries. Request will fail ON the last retry, not after.
   *                   NOTE: passing in Integer.MAX_VALUE will hang the UI
   */
  public ExponentialBackoff(@NonNull Jitter jitter,
                            long delay,
                            long maxDelay,
                            @NonNull TimeUnit timeUnit,
                            int maxRetries) {
    this(null, jitter, delay, maxDelay, timeUnit, maxRetries);
  }

  public ExponentialBackoff(@Nullable String url,
                            @NonNull Jitter jitter,
                            long delay,
                            long maxDelay,
                            @NonNull TimeUnit timeUnit,
                            int maxRetries) {
    Checks.checkNotNull(jitter, "jitter cannot be null");
    Checks.checkArgument(delay >= 0, "delay cannot be less than 0");
    Checks.checkArgument(maxDelay >= 0, "maxDelay cannot be less than 0");
    Checks.checkNotNull(timeUnit, "timeUnit cannot be null");
    Checks.checkArgument(maxRetries >= 0, "maxRetries cannot be less than 0");

    mUrl = url;
    mJitter = jitter;
    mDelay = delay;
    mMaxDelay = maxDelay;
    mTimeUnit = timeUnit;
    // XXX: passing in Integer.MAX_VALUE will hang the UI since RxJava iterates over this number
    mMaxRetries = maxRetries;
  }

  @Override
  public Observable<Long> apply(@NonNull final Observable<? extends Throwable> observable) {
    return observable
      .zipWith(Observable.range(1, mMaxRetries), new BiFunction<Throwable, Integer, Integer>() {
        @Override
        public Integer apply(Throwable throwable, @NonNull Integer retryCount) {
          MaxAdsLog.w(TAG, "Request failed: " + throwable.getMessage()
            + ", retry count: " + retryCount + ", max retries: " + mMaxRetries
            + (!TextUtils.isEmpty(mUrl) ? ", url: " + mUrl : ""));
          return retryCount;
        }
      }).flatMap(new Function<Integer, ObservableSource<Long>>() {
        @Override
        public ObservableSource<Long> apply(Integer attemptNumber) {
          if (attemptNumber == mMaxRetries) {
            final String message = "Request failed: exponential backoff reached max retries: " + mMaxRetries
              + (!TextUtils.isEmpty(mUrl) ? ", url: " + mUrl : "");
            MaxAdsLog.d(TAG, message);
            return Observable.error(new Exception(message));
          }

          long newInterval = getNewInterval(attemptNumber);
          MaxAdsLog.d(TAG, "Retrying request in " + newInterval + " " + mTimeUnit.toString().toLowerCase(Locale.ROOT));
          return Observable.timer(newInterval, mTimeUnit);
        }
      });
  }

  private long getNewInterval(int retryCount) {
    return Math.min((long) (mDelay * Math.pow(retryCount, 2) * mJitter.get()), (long) (mMaxDelay * mJitter.get()));
  }
}

