package io.maxads.ads.base;

import android.support.annotation.MainThread;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;

import io.maxads.ads.base.util.Helpers;
import io.maxads.ads.base.util.MaxAdsLog;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.disposables.Disposable;
import io.reactivex.functions.Consumer;

/**
 * Timer to count down an ad's time until it should be refreshed.
 * Should be paused and then resumed when the app backgrounds and then foregrounds.
 * Calling start while already started will restart the existing timer with the new delay.
 *
 * !!!IMPORTANT!!!
 * This class should never break the chain of ad requests. The refresh timer should always be invoked
 * after an ad loads or fails to load so that the supply of ads continues without the publisher having to manually
 * call load more than once.
 */
@MainThread
public class RefreshTimer {
  @NonNull private static final String TAG = RefreshTimer.class.getSimpleName();

  public static final int DEFAULT_REFRESH_TIME_SECONDS = 60;

  public enum State {
    STARTED,
    PAUSED,
    STOPPED,
    DESTROYED
  }

  public interface Listener {
    void onTimerComplete();
  }

  @NonNull private final Helpers.TimerFactory mTimerFactory;
  @NonNull private State mState;
  @Nullable private Disposable mDisposable;
  @Nullable private Listener mListener;
  // Has the timer ever been called to be started
  private boolean mStartCalled;

  public RefreshTimer() {
    this(new Helpers.TimerFactory());
  }

  @VisibleForTesting
  RefreshTimer(@NonNull Helpers.TimerFactory timerFactory) {
    mTimerFactory = timerFactory;
    mState = State.STOPPED;
  }

  public void setListener(@Nullable Listener listener) {
    mListener = listener;
  }

  /**
   * If the refresh timer is started, then stop the existing timer and restart it with the new delay.
   * If the refresh timer is paused this means that the app is backgrounded so the timer will resume when foregrounded.
   */
  public void start(long delaySeconds) {
    if (mState.equals(State.DESTROYED)) {
      return;
    }

    mStartCalled = true;

   if (mState.equals(State.PAUSED)) {
      return;
   }

    doStart(delaySeconds);
  }

  private void doStart(long delaySeconds) {
    MaxAdsLog.d(TAG, "Starting ad refresh timer for " + delaySeconds + " seconds");

    if (delaySeconds < 0) {
      delaySeconds = DEFAULT_REFRESH_TIME_SECONDS;
    }

    stop();
    mState = State.STARTED;

    // Subscribes on Schedulers.computation() by default
    mDisposable = mTimerFactory.createTimerSeconds(delaySeconds)
      .observeOn(AndroidSchedulers.mainThread())
      .subscribe(new Consumer<Long>() {
        @Override
        public void accept(Long aLong) {
          if (mListener != null) {
            mListener.onTimerComplete();
          }
          stop();
        }
      });
  }

  /**
   * Called when the app is backgrounded.
   * Only calling resume will start the timer again.
   */
  public void pause() {
    if (mState.equals(State.DESTROYED)) {
      return;
    }

    MaxAdsLog.d(TAG, "Pausing ad refresh timer");

    stop();
    mState = State.PAUSED;
  }

  /**
   * Called when the app is foregrounded.
   * If the timer has ever been started then restart the timer with a 0 second delay
   * If the timer was paused without ever starting then put the timer in the stopped state.
   */
  public void resume() {
    if (!mState.equals(State.PAUSED)) {
      return;
    }

    MaxAdsLog.d(TAG, "Resuming ad refresh timer");

    // If the timer was never started convert it back to stopped state so it doesn't start automatically
    if (!mStartCalled) {
      stop();
      return;
    }

    // If the app was backgrounded and then foregrounded, immediately complete the timer to kickoff a new ad request
    doStart(0);
  }

  public void stop() {
    if (mState.equals(State.DESTROYED)) {
      return;
    }

    mState = State.STOPPED;
    if (mDisposable != null) {
      mDisposable.dispose();
    }
  }

  public void destroy() {
    MaxAdsLog.d(TAG, "Destroying ad refresh timer");
    stop();
    mState = State.DESTROYED;
  }

  @Deprecated
  @NonNull
  @VisibleForTesting
  State getState() {
    return mState;
  }

  @Deprecated
  @Nullable
  @VisibleForTesting
  Disposable getDisposable() {
    return mDisposable;
  }
}
