package io.maxads.ads.base;

import android.annotation.SuppressLint;
import android.app.Application;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;

import com.jenzz.appstate.AppState;
import com.jenzz.appstate.adapter.rxjava2.RxAppStateMonitor;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import io.reactivex.functions.Consumer;

public class AppStateManager {
  public interface Listener {
    void onAppStateChanged(AppState appState);
  }

  // AppStateManager is stored globally so use weak references so we don't leak memory in the case that listeners
  // forget to remove themselves
  @NonNull private final List<WeakReference<Listener>> mListeners;

  @SuppressLint("CheckResult")
  public AppStateManager(@NonNull Application application) {
    mListeners = new ArrayList<>();

    RxAppStateMonitor.monitor(application).subscribe(new Consumer<AppState>() {
      @Override
      public void accept(AppState appState) {
        AppStateManager.this.accept(appState);
      }
    });
  }

  public void addListener(@Nullable Listener listener) {
    mListeners.add(new WeakReference<>(listener));
  }

  public void removeListener(@Nullable Listener targetListener) {
    final Iterator<WeakReference<Listener>> iterator = mListeners.iterator();
    while (iterator.hasNext()) {
      final WeakReference<Listener> weakListener = iterator.next();
      final Listener listener = weakListener.get();
      if (listener == null) {
        iterator.remove();
        continue;
      }

      if (listener.equals(targetListener)) {
        iterator.remove();
      }
    }
  }

  @VisibleForTesting
  void accept(AppState appState) {
    final Iterator<WeakReference<Listener>> iterator = mListeners.iterator();
    while (iterator.hasNext()) {
      final WeakReference<Listener> weakListener = iterator.next();
      final Listener listener = weakListener.get();
      if (listener == null) {
        iterator.remove();
        continue;
      }

      listener.onAppStateChanged(appState);
    }
  }

  @Deprecated
  @VisibleForTesting
  @NonNull
  List<WeakReference<Listener>> getListeners() {
    return mListeners;
  }
}
