package io.maxads.ads.banner;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.view.View;

import io.maxads.ads.base.model.Ad;
import io.maxads.ads.base.util.Checks;
import io.maxads.ads.base.util.MaxAdsLog;

public class BannerDecorator implements Banner.Listener {
  @NonNull private static final String TAG = BannerDecorator.class.getSimpleName();

  public interface Listener {
    void onBannerLoaded(@NonNull BannerDecorator bannerDecorator, @NonNull View view);
    void onBannerClicked(@NonNull BannerDecorator bannerDecorator);
    void onBannerError(@NonNull BannerDecorator bannerDecorator);
  }

  @NonNull private final Banner mBanner;
  @NonNull private final Ad mAd;

  private boolean mIsDestroyed;

  @Nullable private Listener mListener;

  public BannerDecorator(@NonNull Banner banner, @NonNull Ad ad) {
    mBanner = banner;
    mBanner.setListener(this);
    mAd = ad;
  }

  public void setListener(@Nullable Listener listener) {
    mListener = listener;
  }

  @NonNull
  public Ad getAd() {
    return mAd;
  }

  public void load() {
    if (!Checks.NoThrow.checkArgument(!isDestroyed(), TAG + " is destroyed")) {
      return;
    }

    MaxAdsLog.d(TAG, "Loading banner for ad unit id: " + mAd.getAdUnitId());
    mAd.trackSelectedUrls();
    mBanner.load();
  }

  public void destroy() {
    MaxAdsLog.d(TAG, "Destroying banner for ad unit id: " + mAd.getAdUnitId());
    mBanner.destroy();
    mIsDestroyed = true;
  }

  public boolean isDestroyed() {
    return mIsDestroyed;
  }

  public void onBannerLoaded(@NonNull Banner banner, @NonNull View view) {
    if (isDestroyed()) {
      return;
    }

    MaxAdsLog.d(TAG, "Banner loaded for ad unit id: " + mAd.getAdUnitId());
    mAd.trackImpressionUrls();
    if (mListener != null) {
      mListener.onBannerLoaded(this, view);
    }
  }

  @Override
  public void onBannerClicked(@NonNull Banner banner) {
    if (isDestroyed()) {
      return;
    }

    MaxAdsLog.d(TAG, "Banner clicked for ad unit id: " + mAd.getAdUnitId());
    mAd.trackClickUrls();
    if (mListener != null) {
      mListener.onBannerClicked(this);
    }
  }

  @Override
  public void onBannerError(@NonNull Banner banner) {
    if (isDestroyed()) {
      return;
    }

    final String errorMessage = "Banner error for ad unit id: " + mAd.getAdUnitId();
    MaxAdsLog.d(TAG, errorMessage);
    mAd.trackError(errorMessage);
    if (mListener != null) {
      mListener.onBannerError(this);
    }
  }
}
