package io.manycore.maven;

import org.apache.http.HttpEntity;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.methods.RequestBuilder;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.HttpMultipartMode;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProject;
import org.json.JSONObject;

import java.io.File;


/**
 * Maven Mojo that uploads jar and pom.xml to manycore server.
 *
 * mvn io.manycore.maven:refrem-maven-plugin:refrem-build
 *
 */
@Mojo(name="refrem-build", defaultPhase = LifecyclePhase.PACKAGE)
public class RefremMavenPlugin extends AbstractMojo {

    /**
     * This is the access token retrieved from https://compiler.refrem.com/upload/
     */
    @Parameter(property = "refrem-build.accessToken")
    private String accessToken;

    /**
     * This is your unique company id retrieved from https://compiler.refrem.com/upload/
     */
    @Parameter(property = "refrem-build.companyId")
    private int companyId;

    /**
     * This is the class in Your jar that contains the main() method.
     */
    @Parameter(property = "refrem-build.mainClass")
    private String mainClass;

    /**
     * This is the upload endpoint, normally https://upload.refrem.com/upload
     */
    @Parameter(property = "refrem-build.uploadEndpoint")
    private String uploadEndpoint;

    /**
     * Points to the current projects so we can find the pom and jar.
     */
    @Parameter(property = "project", readonly = true)
    private MavenProject project;

    public void execute() throws MojoExecutionException, MojoFailureException
    {
        getLog().info("Refrem Maven Plugin is starting...");

        String pomFileName = project.getBasedir().toString() + File.separator + "pom.xml";
        String jarFileName = project.getBuild().getDirectory() + File.separator + project.getBuild().getFinalName() + ".jar";

        getLog().info("Uploading " + pomFileName);
        getLog().info("...and " + jarFileName);
        try (CloseableHttpClient httpclient = HttpClients.createDefault()) {
            HttpEntity data = MultipartEntityBuilder
                    .create()
                    .setMode(HttpMultipartMode.BROWSER_COMPATIBLE)
                    .addBinaryBody("pom", new File(pomFileName), ContentType.APPLICATION_XML, "pom.xml")
                    .addBinaryBody("jar", new File(jarFileName), ContentType.DEFAULT_BINARY, project.getBuild().getFinalName() + ".jar")
                    .addTextBody("cid", "" + companyId)
                    .addTextBody("token", accessToken)
                    .addTextBody("mainClass", mainClass)
                    .build();

            HttpUriRequest request = RequestBuilder.post(uploadEndpoint).setEntity(data).build();

            // Create a custom response handler
            ResponseHandler<String> responseHandler = response -> {
                int status = response.getStatusLine().getStatusCode();
                if (status >= 200 && status < 300) {
                    HttpEntity entity = response.getEntity();
                    return entity != null ? EntityUtils.toString(entity) : null;
                } else {
                    throw new ClientProtocolException("Unexpected response status: " + status);
                }
            };
            String responseBody = httpclient.execute(request, responseHandler);
            JSONObject obj = new JSONObject(responseBody);
            if (obj.getBoolean("success")) {
                getLog().info("Successfully uploaded jar+pom to Refrem: " + obj.getString("message"));
            } else {
                getLog().error("Failed to upload jar+pom to Refrem: " + obj.getString("message"));
            }
        }
        catch (Exception e) {
            getLog().error("Error: " + e.toString());
        }
    }

}
