package io.magus.methodmap.json;

import io.magus.methodmap.ArgsFactory;
import io.magus.methodmap.MethodHandleInfo;
import io.magus.methodmap.error.ArgsProductionException;

import java.io.IOException;
import java.lang.reflect.Type;

import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;

/**
 * ArgsFactory that converts a JSON array to an array of java objects which coincide with the types
 * of the expected parameters of the target method. Uses the Jackson JSON library to perform the
 * necessary conversions.
 *
 * @author Enseart Simpson
 *
 */
public class JacksonArgsFactory implements ArgsFactory {

	private final ObjectMapper objectMapper;
	private final ArrayNode jsonArgsArray;

	/**
	 * @param objectMapper	the objectmapper that will be used to perform the conversions
	 * @param jsonArgsArray	an array of Jackson JSON objects that are to be converted into the
	 * 						types expected by the method
	 *
	 */
	public JacksonArgsFactory(ObjectMapper objectMapper, ArrayNode jsonArgsArray) {
		this.objectMapper = objectMapper;
		this.jsonArgsArray = jsonArgsArray;
	}

	/**
	 * @param objectMapper			the objectmapper that will be used to perform the conversions
	 * @param jsonArrayString		a JSON array formatted string that will be converted to an array
	 * 								of objects containing the types expected by the method
	 * @throws JsonParseException
	 * @throws JsonMappingException
	 * @throws IOException
	 */
	public JacksonArgsFactory(ObjectMapper objectMapper, String jsonArrayString)
			throws JsonParseException, JsonMappingException, IOException {
		this(objectMapper, objectMapper.readValue(jsonArrayString, ArrayNode.class));
	}

	@Override
	public Object[] produceArgs(MethodHandleInfo methodHandleInfo)
			throws ArgsProductionException {
		Type[] parameterTypes;
		Object arg, args[];
		int i, numJsonArgs;
		JavaType javaType;

		parameterTypes = methodHandleInfo.getMethod().getGenericParameterTypes();

		args = new Object[parameterTypes.length];
		numJsonArgs = jsonArgsArray.size();

		for(i = 0; i < parameterTypes.length; i++) {
			javaType = objectMapper.constructType(parameterTypes[i]);

			arg = (i < numJsonArgs) ?
					objectMapper.convertValue(jsonArgsArray.get(i), javaType) : null;

			args[i] = arg;
		}

		return args;
	}

}
